/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#define _BSD_SOURCE /* lstat() */

#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include "main.h"
#include "cdw_config.h"
#include "cdw_task.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_burn_disc.h"
#include "cdw_write_wizard.h"

#include "cdw_string.h"
#include "cdw_utils.h"
#include "cdw_drive.h"
#include "cdw_file_picker.h"
#include "cdw_file_manager.h"
#include "cdw_cdrecord.h"
#include "cdw_growisofs.h"
#include "cdw_xorriso.h"
#include "cdw_digest.h"
#include "cdw_logging.h"
#include "cdw_main_window.h"
#include "cdw_iso9660.h"
#include "cdw_ext_tools.h"
#include "cdw_config_window.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"
#include "cdw_calculate_digest.h"


static cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *parent_task, cdw_disc_t *disc);
static bool cdw_burn_disc_is_proper(cdw_disc_t *disc);
static bool cdw_burn_disc_get_source(cdw_task_t *task);
static void cdw_burn_disc_release_source(cdw_task_t *task);
static bool cdw_burn_disc_check_wodim_large_image(cdw_task_t *task);
static bool cdw_burn_disc_check_overwrite_content(cdw_task_t *task, cdw_disc_t *disc);

/**
   \brief Check if everything is ready for writing to optical disc and then perform it

   Function does pre-writing checks, displays write wizard and
   calls burn_dispatcher() to do the job.

   Supported tasks are writing directly to disc and writing image to disc.
   Supported disc types are data cd and dvd.
   When combined the two things above we have following tasks:
    - write files to dvd
    - write image to dvd
    - write files to cd
    - write image to cd

   The decision is based on value of task->source and type of
   disc. Disc type is checked during this function call, before performing
   task.

   Function checks if image is available or if any files are selected for
   writing.

   \param task_id CDW_TASK_BURN_FROM_IMAGE or CDW_TASK_BURN_FROM_FILES

   \return CDW_OK on success
   \return CDW_NO if some preconditions were not met
   \return CDW_CANCEL if user cancels writing
   \return CDW_ERROR on failure or some error
*/
cdw_rv_t cdw_burn_disc(int task_id)
{
	cdw_assert (task_id == CDW_TASK_BURN_FROM_FILES || task_id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: called the function with incorrect task id %d\n", task_id);

	cdw_disc_t *disc = cdw_disc_new();
	cdw_rv_t crv = cdw_disc_get(disc);
	if (crv != CDW_OK) {
		cdw_disc_delete(&disc);
		return crv;
	}

	/* check if the disc in drive is of proper type, and in proper state */
	if (!cdw_burn_disc_is_proper(disc)) {
		cdw_disc_delete(&disc);
		return CDW_NO;
	}

	/* Update information about disc type, size and usage. */
	cdw_main_window_disc_info_view_display_data(disc);
	cdw_main_window_volume_info_view_update(disc->capacity.sectors_used, disc->capacity.sectors_total, true);


	cdw_task_t *task = cdw_task_new(task_id, disc);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		cdw_disc_delete(&disc);
		return CDW_NO;
	}

	if (!cdw_burn_disc_check_wodim_large_image(task)) {
		cdw_burn_disc_release_source(task);
		cdw_disc_delete(&disc);
		cdw_task_delete(&task);
		return CDW_CANCEL;
	}

	if (!cdw_burn_disc_get_source(task)) {
		cdw_disc_delete(&disc);
		cdw_task_delete(&task);
		return CDW_NO;
	}

	cdw_rv_t decision = cdw_write_wizard(task, disc);

	if (decision != CDW_OK) {
		cdw_burn_disc_release_source(task);
		cdw_disc_delete(&disc);
		cdw_task_delete(&task);
		if (decision == CDW_ERROR) {
			/* 2TRANS: this is the title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is the message in the dialog window */
					   _("cdw can't create write wizard. Please restart cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}
		return decision;
	}


	if (!cdw_burn_disc_check_overwrite_content(task, disc)) {
		/* there was already some content on the disc, it might have
		   been overwritten, but user decided not to overwrite it */
		cdw_burn_disc_release_source(task);
		cdw_disc_delete(&disc);
		cdw_task_delete(&task);
		return CDW_CANCEL;
	}

	cdw_main_window_wrefresh();

	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: task id has changed\n");

	/* 2TRANS: this is a title of process window;
	   "disc" is "optical disc" */
	cdw_processwin_create(_("Writing to disc"),
			      /* 2TRANS: this is message in process window;
				 "disc" is "optical disc" */
			      _("Writing to optical disc"), true);

	/* burn_dispatcher() calls actual function performing burning */
	cdw_rv_t command_return = cdw_burn_disc_dispatcher(task, disc);
	/* writing has been finished, erase information that was updated
	   (and up to date) only during actual writing; displaying it
	   after this point may be misleading - user might think that
	   it is somehow still valid */
	cdw_processwin_erase_fifo_and_speed();
	cdw_processwin_erase_eta();
	cdw_burn_disc_release_source(task);
	/* FIXME: always call cdw_disc_delete() at ejecting a tray, otherwise
	   any subsequent calls to cdio library may return invalid results;
	   most probably 'disc' should be passed to cdw_drive_eject*() */
	cdw_disc_delete(&disc);

	const char *drive_fullpath = cdw_drive_get_drive_fullpath();
	cdw_drive_eject_tray_with_ui_update(drive_fullpath);

	cdw_rv_t tool_status = cdw_task_check_tool_status(task);

	if (command_return == CDW_OK && tool_status == CDW_OK) {

		/* Keep in mind that trying to verify a dummy write is
		   a bad idea. */
		if (task->burn.verify && !task->burn.dummy) {
			cdw_vdm ("INFO: \"verify burn\" is true, \"dummy write\" is false, attempting verification\n");
			cdw_drive_close_tray_with_ui_update(drive_fullpath);

			cdw_rv_t verification = cdw_calculate_digest(CDW_CALCULATE_DIGEST_MODE_DISC_FILE, task->image_file_fullpath);
			if (verification != CDW_OK) {
				cdw_vdm ("ERROR: verification function returns !CDW_OK\n");
				command_return = CDW_ERROR;
			}
		} else {
			cdw_vdm ("INFO: \"verify burn\" is false or \"dummy write\" is true, not attempting verification\n");
		}
	} else {
		/* incorrect situation, already covered by debug messages above */
		command_return = CDW_NO;
	}
	int status = cdw_drive_status(drive_fullpath);
	if (task->burn.eject) {
		if (status != CDW_CD_TRAY_OPEN) {
			cdw_drive_eject_tray_with_ui_update(drive_fullpath);
		}
	} else {
		if (status == CDW_CD_TRAY_OPEN) {
			cdw_drive_close_tray_with_ui_update(drive_fullpath);
		}
	}

	cdw_rv_t retval = CDW_OK;
	if (command_return == CDW_OK && tool_status == CDW_OK) {
		cdw_task_save_options(task);

		/* 2TRANS: this is message in dialog window:
		   operation finished with unknown result, probably success */
		cdw_processwin_destroy(_("Writing finished"), true);
		retval = CDW_OK;
	} else {
		cdw_vdm ("ERROR: returning CDW_ERROR due to errors reported above\n");
		/* 2TRANS: this is message in dialog window:
		   operation finished (most probably) with error */
		cdw_processwin_destroy(_("Writing probably failed"), true);
		retval = CDW_ERROR;
	}

	/* 2TRANS: this is the title of the dialog window, displaying messages
	   from the program writing iso image or selected files to CD disc */
	cdw_logging_display_log_conditional(_("\"Write\" log"));


	cdw_task_delete(&task);
	return retval;
}





bool cdw_burn_disc_check_wodim_large_image(cdw_task_t *task)
{
	bool ask = task->id == CDW_TASK_BURN_FROM_IMAGE
		&& task->burn.tool.id == CDW_TOOL_CDRECORD
		&& cdw_ext_tools_is_cdrecord_wodim();

	if (!ask) {
		return true;
	}

	/* sector = 2048 bytes */
	long long sectors = cdw_iso_image_get_n_sectors();
	double gigabytes = ((float) sectors * 2.0) / (1024.0 * 1024.0);
	cdw_vdm ("INFO: sectors = %lld -> %fGB\n", sectors, gigabytes);
	if (sectors > 2 * 1024 * 1024) {
		cdw_vdm ("INFO: large image: %lld sectors -> %fGB\n", sectors, gigabytes);
		/* 2TRANS: this is the title of dialog window */
		cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
						  /* 2TRANS: this is the message in the dialog window */
						  _("You are attempting to use wodim to burn large image. This task will probably fail. You should try using another tool (cdrecord or growisofs) for this task. Do you want to continue?"),
						  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_ERROR);
		if (crv == CDW_OK) {
			/* 2TRANS: this is message printed in log file;
			   %ld is size of ISO9660 image (in sectors) */
			cdw_logging_write(_("Attempting to burn ISO image of size %ld sectors with wodim, this may fail...\n"), sectors);
			return true;
		} else {
			return false;
		}
	} else {
		return true;
	}
}





bool cdw_burn_disc_check_overwrite_content(cdw_task_t *task, cdw_disc_t *disc)
{
	bool ask = disc->state_empty != CDW_TRUE
		&& (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI
		    || task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE);

	if (!ask) {
		return true;
	}

	/* 2TRANS: this is the title of dialog window */
	cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
					  /* 2TRANS: this is the message in the dialog window */
					  _("The disc is not blank, you will overwrite its current content. Continue?"),
					  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
	if (crv == CDW_OK) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Pass control to cdrecord, growisofs or xorriso code to perform burning

   This function is just a dispatcher, and calls cdw_cdrecord_run_task(),
   cdw_growisofs_run_task() or cdw_xorriso_run_task(), depending on
   task->burn.tool.id.

   The function checks all important assertions before doing actual
   dispatching, but only in debug build.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return return value from cdrecord, growisofs or xorriso cdw_*_run_task() function
*/
cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->burn.tool.id != CDW_TOOL_XORRISO) {
		cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: calling the function with uninitialized disc mode\n");
	}
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: calling the function with uninitialized session mode\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);
	cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD
		    || task->burn.tool.id == CDW_TOOL_GROWISOFS
		    || task->burn.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: incorrect task->main_tool_id %lld\n", task->burn.tool.id);

	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window:
			   writing selected files to CD is in progress */
			cdw_processwin_display_main_info(_("Writing files to CD..."));
		} else {
			/* 2TRANS: this is message in process window:
			   writing selected files to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing files to DVD..."));
		}
	} else {
		if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window -
			   writing iso image to CD is in progress */
			cdw_processwin_display_main_info(_("Writing image to CD..."));
		} else {
			/* 2TRANS: this is message in process window -
			   writing iso image to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing image to DVD..."));
		}
	}

	/* at this point we don't care what kind of disc we have,
	   because "external tools" module already checked disc type and
	   decided which tool to use, e.g. to use cdrecord or growisofs for
	   writing to DVD; so we dispatch basing only on tool id */
	cdw_rv_t crv = CDW_OK;
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_vdm ("INFO: dispatching \"burn\" task to cdrecord\n");
		crv = cdw_cdrecord_run_task(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_vdm ("INFO: dispatching \"burn\" task to growisofs\n");
		crv = cdw_growisofs_run_task(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_vdm ("INFO: dispatching \"burn\" task to xorriso\n");
		crv = cdw_xorriso_run_task(task, disc);
	} else {
		cdw_assert (0, "ERROR: wrong tool id '%lld' for the job\n", task->burn.tool.id);
	}

	/* it is too early to check task->tool_status.ok */
	cdw_vdm ("INFO: burn dispatcher returns %s\n", cdw_utils_get_crv_label(crv));
	return crv;
}





bool cdw_burn_disc_get_source(cdw_task_t *task)
{
	bool source_available = false;

	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		/* 2TRANS: this is the title of dialog window */
		cdw_rv_t crv = cdw_file_picker(_("Path to iso image"),
					       /* 2TRANS: this is the message in the dialog window;
						  "writing" means writing to optical disc */
					       _("Please enter FULL path to an existing ISO image file for writing:"),
					       &(task->image_file_fullpath),
					       CDW_FS_FILE, R_OK, CDW_FS_EXISTING);

		cdw_main_window_wrefresh();
		if (crv == CDW_OK) {
			/* TODO: replace with call to generic
			   cdw_fs_get_size_(const char *fullpath, int units); */
			task->burn.data_size_mb = cdw_iso_image_get_size_mb(task->image_file_fullpath);
			source_available = true;
		}
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {

		/* 2TRANS: this is title of dialog window */
		cdw_processwin_create(_("Writing to disc"), "", false);
		
		/* 2TRANS: this is message in dialog window - cdw is putting file names of selected files on a list. */
		cdw_processwin_display_sub_info(_("Preparing list of files..."));

		cdw_rv_t crv = cdw_file_manager_create_graftpoints_file();

		cdw_processwin_destroy("", false);

		if (crv == CDW_OK) {
			long long size = cdw_selected_files_get_size();
			task->burn.data_size_mb = (((double) size) / 1024.0) / 1024.0;

			source_available = true;
			if (cdw_selected_files_file_over_4gb_present()) {
				cdw_vdm ("WARNING: selected file > 4 GB\n");
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Warning"),
						   /* 2TRANS: this is the message in the dialog window:
						      disc is not recognized as supported by
						      currently selected tools */
						   _("One of selected files has a size over 4GB. Choose your tools wisely (Configuration -> Tools), otherwise writing to disc will give incorrect results."),
						   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			} else {
				cdw_sdm ("INFO: file size <= 4 GB\n");
			}
		} else if (crv == CDW_NO) {
			; /* probably no selected files */
		} else {
			cdw_vdm ("ERROR: failed to correctly create graftpoints file\n");
		}
	} else {
		; /* covered by caller's assert */
	}

	return source_available;
}





void cdw_burn_disc_release_source(cdw_task_t *task)
{
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_file_manager_delete_graftpoints_file();
	}

	return;
}





bool cdw_burn_disc_is_proper(cdw_disc_t *disc)
{
	if (disc->type == CDW_DISC_TYPE_UNKNOWN) {
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window:
				      disc is not recognized as supported by
				      currently selected tools */
				   _("Can't recognize disc in drive. Try changing tools family in Configuration -> Tools or use different disc type."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}

	if (disc->type_writable != CDW_TRUE) {
		/* CD-AUDIO, DVD-ROM etc. */
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - disc is not writable */
				   _("Disc in drive is read-only."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}


	if (disc->state_writable == CDW_UNKNOWN) {
		/* perhaps device path is invalid? */

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window: for some
				      reason cdw cannot read disc metainfo; this may be a
				      problem with the hardware configuration or with the family
				      of tools (cdrtools / dvd+rw-tools) used for the task */
				   _("Cannot get full information about disc. Please check configuration of hardware or tools. (Are you using wodim?)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else if (disc->state_writable == CDW_FALSE) {

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - user tries to write to a closed disc */
				   _("Cannot write to this disc anymore, disc is closed."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else {
		/* disc->state_appendable == CDW_DISC_APPENDABLE_YES */
		if (disc->type == CDW_DVD_RP_DL
		    && cdw_config_support_dvd_rp_dl()) {

			/* 2TRANS: this is the title of the dialog window */
			cdw_rv_t crv2 = cdw_buttons_dialog(_("Information"),
							   /* 2TRANS: this is the message in the dialog window */
							  _("You are attempting to burn to a DVD+R DL disc. You REALLY need to use high quality discs and have a bit of luck to correctly burn data to DVD+R DL. Additionally cdw offers very limited support for DVD+RD DL discs. Continue?"),
							  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_DIALOG);
			if (crv2 == CDW_OK) {
				return true;
			} else {
				return false;
			}
		}
		return true;
	}
}





cdw_rv_t cdw_write_copy(cdw_write_t *target, cdw_write_t *source)
{
	target->tool.id    = source->tool.id;
	/* 'label' is a pointer to const label in cdw_ext_tools. Don't
	   use string copying functions. */
	target->tool.label = source->tool.label;


	for (int i = 0; i < CDW_SESSION_MODE_N_MAX; i++) {
		target->session_modes[i] = source->session_modes[i];
	}
	target->session_modes_n_labels   = source->session_modes_n_labels;
	target->session_mode             = source->session_mode;


	for (int i = 0; i < CDW_DISC_MODE_N_MODES; i++) {
		target->disc_modes[i] = source->disc_modes[i];
	}
	target->disc_modes_n_labels   = source->disc_modes_n_labels;
	target->disc_mode             = source->disc_mode;


	target->speed_id = source->speed_id;
	target->verify   = source->verify;
	target->dummy    = source->dummy;
	target->eject    = source->eject;


	target->data_size_mb = source->data_size_mb;


	target->cdrecord_burnproof = source->cdrecord_burnproof;
	target->cdrecord_pad       = source->cdrecord_pad;
	target->cdrecord_pad_size  = source->cdrecord_pad_size;


	cdw_rv_t crv = cdw_string_set(&(target->cdrecord_other_options), source->cdrecord_other_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set cdrecord_other_options from src = \"%s\"\n", source->cdrecord_other_options);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(target->xorriso_burn_other_options), source->xorriso_burn_other_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set xorriso_burn_other_options from src = \"%s\"\n", source->xorriso_burn_other_options);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(target->growisofs_other_options), source->growisofs_other_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set growisofs_other_options from src = \"%s\"\n", source->growisofs_other_options);
		return CDW_ERROR;
	}

	return CDW_OK;
}





void cdw_write_init(cdw_write_t *write)
{
	write->tool.id = CDW_TOOL_NONE;
	write->tool.label = (char *) NULL;

	write->cdrecord_other_options = (char *) NULL;
	write->xorriso_burn_other_options = (char *) NULL;
	write->growisofs_other_options = (char *) NULL;

	return;
}





void cdw_write_clean(cdw_write_t *write)
{
	if (write->cdrecord_other_options) {
		free(write->cdrecord_other_options);
		write->cdrecord_other_options = (char *) NULL;
	}
	if (write->xorriso_burn_other_options) {
		free(write->xorriso_burn_other_options);
		write->xorriso_burn_other_options = (char *) NULL;
	}
	if (write->growisofs_other_options) {
		free(write->growisofs_other_options);
		write->growisofs_other_options = (char *) NULL;
	}

	return;
}





cdw_rv_t cdw_write_set_defaults(cdw_write_t *write)
{
	for (int i = 0; i < CDW_SESSION_MODE_N_MAX; i++) {
		write->session_modes[i] = CDW_SESSION_MODE_INIT;
	}
	write->session_modes_n_labels   = 0;
	write->session_mode             = CDW_SESSION_MODE_INIT;


	for (int i = 0; i < CDW_DISC_MODE_N_MODES; i++) {
		write->disc_modes[i] = CDW_DISC_MODE_INIT;
	}
	write->disc_modes_n_labels   = 0;
	write->disc_mode             = CDW_DISC_MODE_INIT;


	write->speed_id = 0;
	write->verify   = true;
	write->dummy    = false;
	write->eject    = false;  /* Keep consistent with cdw_erase_t->eject. */


	write->data_size_mb = 0.0;


	write->cdrecord_burnproof = true;
	write->cdrecord_pad = true;
	/* the "150" value is selected after small tests with DVD; originally it was 63, which
	   worked just fine for CDs, but for DVDs it was insufficient;
	   TODO: the same value is used in code creating configuration forms,
	   so it should be a constant defined in some header */
	/* OLD COMMENT: value of 63 is taken from here:
	   http://www.troubleshooters.com/linux/coasterless.htm */
	write->cdrecord_pad_size = 150;


	/* Can't set to (char *) NULL because these options are used
	   as argument of concat() in some places. */
	cdw_rv_t crv = cdw_string_set(&(write->cdrecord_other_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set cdrecord_other_options\n");
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(write->xorriso_burn_other_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set xorriso_burn_other_options\n");
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(write->growisofs_other_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set growisofs_other_options\n");
		return CDW_ERROR;
	}

	return CDW_OK;
}





void cdw_write_debug_print_options(cdw_write_t *write)
{
	cdw_vdm ("INFO: Write options:\n");

	cdw_vdm ("INFO:                    tool ID = %lld (%s)\n",  write->tool.id, cdw_ext_tools_get_tool_name(write->tool.id));
	cdw_vdm ("INFO:                 tool label = %s\n",         write->tool.label);

	cdw_vdm ("INFO:               session mode = %lld\n",       write->session_mode);
	cdw_vdm ("INFO:                  disc mode = %lld\n",       write->disc_mode);

	cdw_vdm ("INFO:                   speed ID = %lld\n",       write->speed_id);

	cdw_vdm ("INFO:                     verify = %s\n",         write->verify ? "true" : "false");
	cdw_vdm ("INFO:                      dummy = %s\n",         write->dummy ? "true" : "false");
	cdw_vdm ("INFO:                      eject = %s\n",         write->eject ? "true" : "false");

	cdw_vdm ("INFO:                  data size = %g MB\n",      write->data_size_mb);

	cdw_vdm ("INFO:         cdrecord burnproof = %s\n",         write->cdrecord_burnproof ? "true" : "false");
	cdw_vdm ("INFO:               cdrecord pad = %s\n",         write->cdrecord_pad ? "true" : "false");
	cdw_vdm ("INFO:          cdrecord pad size = %d\n",         write->cdrecord_pad_size);

	cdw_vdm ("INFO:     cdrecord other options = \"%s\"\n",     write->cdrecord_other_options);
	cdw_vdm ("INFO: xorriso other burn options = \"%s\"\n\n",   write->xorriso_burn_other_options);
	cdw_vdm ("INFO:    growisofs other options = \"%s\"\n",     write->growisofs_other_options);

	return;
}
