/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * xine network related stuff
 *
 */
/* required for getsubopt(); the __sun test avoids compilation problems on
    solaris. On FreeBSD defining this disable BSD functions to be visible
    and remove INADDR_NONE */
#if ! defined (__sun__) && ! defined (__OpenBSD__)  && ! defined(__FreeBSD__) && ! defined(__NetBSD__) && ! defined(__APPLE__) && ! defined (__DragonFly__)
#define _XOPEN_SOURCE 500
#endif
/* required for strncasecmp() */
#define _BSD_SOURCE 1
#define _DEFAULT_SOURCE 1
/* required to enable POSIX variant of getpwuid_r on solaris */
#define _POSIX_PTHREAD_SEMANTICS 1

//#warning IMPLEMENT POST SUPPORT

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_READLINE

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#if !defined(__hpux)
#include <string.h>
#endif
#include <stdarg.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pthread.h>

#include <readline.h>
#include <history.h>

#include "common.h"
#include "network.h"

#ifndef	MSG_NOSIGNAL
#define	MSG_NOSIGNAL	     0
#endif

#define DEFAULT_XINECTL_PORT "6789"
#define DEFAULT_SERVER       "localhost"

#define PROGNAME             "xine-remote"
#define PROGVERSION          "0.1.2"

#define MAX_LINE 20480
#define MAX_ARGS 256

#define COMMANDS_PREFIX      "/\377\200COMMANDS"

static const uint8_t tab_char_type[256] = {
    [0]    = 1,
    ['\r'] = 2,
    ['\n'] = 2,
    [';']  = 2,
    ['\t'] = 4,
    [' ']  = 8,
    ['"']  = 16,
    ['\''] = 32,
    ['\\'] = 64
};

static int sock_create(const char *service, const char *transport, struct sockaddr_in *sin) {
  struct servent    *iservice;
  struct protoent *itransport;
  int                sock;
  int                type;
  int                proto = 0;

  memset(sin, 0, sizeof(*sin));

  sin->sin_family = AF_INET;
  sin->sin_port   = htons(atoi(service));

  if(!sin->sin_port) {
    iservice = getservbyname(service, "tcp");

    if(!iservice)
      fprintf (stderr, "Service not registered: %s\n", service);
    else
      sin->sin_port = iservice->s_port;
  }

  itransport = getprotobyname(transport);

  if(!itransport)
    fprintf (stderr, "Protocol not registered: %s\n", transport);
  else
    proto = itransport->p_proto;

  if(!strcmp(transport, "udp"))
    type = SOCK_DGRAM;
  else
    type = SOCK_STREAM;

  sock = socket(AF_INET, type, proto);

  if(sock < 0) {
    fprintf (stderr, "Cannot create socket: %s\n", strerror(errno));
    return -1;
  }

  if (fcntl(sock, F_SETFD, FD_CLOEXEC) < 0) {
    fprintf (stderr, "** socket cannot be made uninheritable (%s)\n", strerror(errno));
  }

  return sock;
}

/*
 * Check for socket validity.
 */
static int sock_check_opened(int socket) {
  for (;;) {
    fd_set exceptfds;
    struct timeval tv = { 0, 0 };
    int    retval;

    FD_ZERO (&exceptfds);
    FD_SET (socket, &exceptfds);
    retval = select (socket + 1, NULL, NULL, &exceptfds, &tv);

    if (retval == -1) {
      if ((errno != EAGAIN) && (errno != EINTR))
        return 0;
    } else {
      return 1;
    }
  }
}

/*
 * Write to socket.
 */
static int _sock_write(int socket, const char *buf, int len) {
  int _len = len;

  if (socket < 0)
    return -1;

  if (!sock_check_opened (socket))
    return -1;

  while (_len > 0) {
    ssize_t size = write (socket, buf, _len);
    if (size <= 0)
      return -1;
    _len -= size;
    buf += size;
  }

  return len - _len;
}

#ifdef NETWORK_CLIENT
/** client only *********************************************************/

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

#ifdef HAVE_GETOPT_LONG
#  include <getopt.h>
#else
#  include "getopt.h"
#endif

/* options args */
static const char short_options[] = "?hH:P:ncv";
static const struct option long_options[] = {
  { "help"           , no_argument      , 0, 'h' },
  { "host"           , required_argument, 0, 'H' },
  { "port"           , required_argument, 0, 'P' },
  { "command"        , no_argument      , 0, 'c' },
  { "noconnect"      , no_argument      , 0, 'n' },
  { "version"        , no_argument      , 0, 'v' },
  { 0                , no_argument      , 0,  0  }
};


typedef struct session_s session_t;
typedef struct session_commands_s session_commands_t;
typedef void (*client_func_t)(session_t *, session_commands_t *, const char *);

static void client_noop(session_t *, session_commands_t *, const char *);
static void client_help(session_t *, session_commands_t *, const char *);
static void client_version(session_t *, session_commands_t *, const char *);
static void client_open(session_t *, session_commands_t *, const char *);
static void client_close(session_t *, session_commands_t *, const char *);
static void client_quit(session_t *, session_commands_t *, const char *);

struct session_s {
  char             host[256];
  char             port[256];
  int              socket;
  int              console;
  int              running;
  char             prompt[544];
  pthread_t        thread;
  pthread_mutex_t  console_mutex;
};

#define ORIGIN_SERVER 1
#define ORIGIN_CLIENT 2

struct session_commands_s {
  char           *command;
  int             origin;
  int             enable;
  client_func_t   function;
};

typedef struct client_commands_s {
  const char     *command;
  int             origin;
  int             enable;
  client_func_t   function;
} client_commands_t;

static session_t             session;
static const client_commands_t client_commands[] = {
  { "?",           ORIGIN_CLIENT,   1, client_help    },
  { "version",     ORIGIN_CLIENT,   1, client_version },
  { "open",        ORIGIN_CLIENT,   1, client_open    },
  { "close",       ORIGIN_CLIENT,   1, client_close   },
  { "quit",        ORIGIN_CLIENT,   1, client_quit    },
  { NULL,          ORIGIN_CLIENT,   1, NULL           }
};

static session_commands_t  **session_commands = NULL;

static int __attribute__ ((format (printf, 3, 4))) __sock_write(int socket, int cr, const char *msg, ...) {
  char     buf[MAX_LINE];
  size_t   s;
  va_list  args;

  va_start(args, msg);
  s = vsnprintf (buf, MAX_LINE - 1, msg, args);
  va_end(args);

  /* Each line sent is '\n' terminated */
  if(cr) {
    if (s && (buf [s - 1] != '\n')) {
      buf[s++] = '\n';
      buf[s] = 0;
    }
  }

  return _sock_write (socket, buf, s);
}

static int sock_client(const char *host, const char *service, const char *transport) {
  union {
    struct sockaddr_in in;
    struct sockaddr sa;
  } fsin;
  struct hostent      *ihost;
  int                  sock;

  sock = sock_create(service, transport, &fsin.in);
  if (sock < 0) {
    return -1;
  }

  if ((fsin.in.sin_addr.s_addr = inet_addr(host)) == INADDR_NONE) {
    ihost = gethostbyname(host);

    if(!ihost) {
      fprintf (stderr, "Unknown host: %s\n", host);
      return -1;
    }
    memcpy(&fsin.in.sin_addr, ihost->h_addr_list[0], ihost->h_length);
  }

  if(connect(sock, &fsin.sa, sizeof(fsin.in)) < 0) {
    int err = errno;

    close(sock);
    errno = err;
    fprintf (stderr, "Unable to connect %s[%s]: %s\n", host, service, strerror(errno));
    return -1;
  }

  return sock;
}

#if __GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 91)
#define write_to_console_unlocked(session, msg, args...)  __sock_write(session->console, 1, msg, ##args)
#define write_to_console_unlocked_nocr(session, msg, args...) __sock_write(session->console, 0, msg, ##args)
#else
#define write_to_console_unlocked(session, ...)  __sock_write(session->console, 1, __VA_ARGS__)
#define write_to_console_unlocked_nocr(session, ...) __sock_write(session->console, 0, __VA_ARGS__)
#endif

static int __attribute__ ((format (printf, 2, 3))) write_to_console(session_t *session, const char *msg, ...) {
  char     buf[MAX_LINE];
  va_list  args;
  int      err;

  va_start(args, msg);
  vsnprintf(buf, MAX_LINE, msg, args);
  va_end(args);

  pthread_mutex_lock(&session->console_mutex);
  err = write_to_console_unlocked(session, "%s", buf);
  pthread_mutex_unlock(&session->console_mutex);

  return err;
}

#if 0
static int __attribute__ ((format (printf, 2, 3)) write_to_console_nocr(session_t *session, const char *msg, ...) {
  char     buf[MAX_LINE];
  va_list  args;
  int      err;

  va_start(args, msg);
  vsnprintf(buf, MAX_LINE, msg, args);
  va_end(args);

  pthread_mutex_lock(&session->console_mutex);
  err = write_to_console_unlocked_nocr(session, buf);
  pthread_mutex_unlock(&session->console_mutex);

  return err;
}
#endif

static void session_update_prompt(session_t *session) {
  if(session == NULL)
    return;

  if(session->socket >= 0)
    snprintf(session->prompt, sizeof(session->prompt), "[%s:%s]"PROGNAME" >", session->host, session->port);
  else
    strlcpy(session->prompt, "[******:****]"PROGNAME" >", sizeof(session->prompt));
}

static void session_create_commands(session_t *session) {
  static const size_t num_commands = (sizeof(client_commands) / sizeof(client_commands[0]));
  int i;

  if(session == NULL)
    return;

  if(session_commands != NULL) {
    i = 0;
    while(session_commands[i]->command != NULL) {
      free(session_commands[i]->command);
      free(session_commands[i]);
      i++;
    }
    free(session_commands[i]);
    free(session_commands);
  }

  session_commands = (session_commands_t **) calloc(num_commands, sizeof(session_commands_t *));
  for(i = 0; client_commands[i].command != NULL; i++) {
    session_commands[i] = (session_commands_t *) malloc(sizeof(session_commands_t));
    session_commands[i]->command   = strdup(client_commands[i].command);
    session_commands[i]->origin    = client_commands[i].origin;
    session_commands[i]->enable    = client_commands[i].enable;
    session_commands[i]->function  = client_commands[i].function;
  }
  session_commands[i] = (session_commands_t *) malloc(sizeof(session_commands_t));
  session_commands[i]->command   = NULL;
  session_commands[i]->origin    = ORIGIN_CLIENT;
  session_commands[i]->enable    = 0;
  session_commands[i]->function  = NULL;

}

/*
 * Client commands
 */
static void client_noop(session_t *session, session_commands_t *command, const char *cmd) {
  (void)session;
  (void)command;
  (void)cmd;
}

static size_t _list_sess_cmds (char *buf, size_t bsize, int line_width) {
  char *p = buf, *s = buf, *e = buf + bsize - 1;
  int cwidth = 0, cnum, col = 0, i;

  for (i = 0; session_commands[i]->command; i++) {
    if (session_commands[i]->enable) {
      size_t l = strlen (session_commands[i]->command);

      if ((int)l > cwidth)
        cwidth = l;
    }
  }
  cwidth += 1;
  cnum = (line_width - 8) / cwidth;
  if (cnum <= 0)
    cnum = 1;

  for (i = 0; session_commands[i]->command; i++) {
    if (session_commands[i]->enable) {
      size_t l = strlen (session_commands[i]->command);
  
      if (col == 0) {
        if (p + 8 > e)
          break;
        memset (p, ' ', 8); p += 8;
      }
      if (p + cwidth > e)
        break;
      memcpy (p, session_commands[i]->command, l); p += l;
      s = p;
      if (++col < cnum) {
        memset (p, ' ', cwidth - l); p += cwidth - l;
      } else {
        if (p >= e)
          break;
        *p++ = '\n';
        col = 0;
      }
    }
  }
  p = s;
  if (p < e)
    *p++ = '\n';
  *p = 0;
  return p - buf;
}

static void client_help(session_t *session, session_commands_t *command, const char *cmd) {
  char buf[MAX_LINE], *p = buf, *e = buf + sizeof (buf);

  (void)cmd;
  if((session == NULL) || (command == NULL))
    return;

  p += strlcpy (p, "Available commands are:\n", e - p);
  /* if (p > e) p = e; */
  _list_sess_cmds (p, e - p, 80);
  write_to_console(session, "%s\n", buf);
}

static void client_version(session_t *session, session_commands_t *command, const char *cmd) {

  (void)cmd;
  if((session == NULL) || (command == NULL))
    return;

  write_to_console(session, "%s version %s\n\n", PROGNAME, PROGVERSION);
}

static void client_close (session_t *session, session_commands_t *command, const char *cmd) {
  (void)cmd;
  if((session == NULL) || (command == NULL))
    return;

  if (session->socket >= 0) {
    int i = 0;

    _sock_write (session->socket, "exit\n", 5);
    close (session->socket);
    pthread_mutex_lock (&session->console_mutex);
    session->socket = -1;
    pthread_mutex_unlock (&session->console_mutex);
    session_update_prompt (session);

    while(session_commands[i]->command != NULL) {
      if(session_commands[i]->origin == ORIGIN_SERVER)
	session_commands[i]->enable = 0;
      i++;
    }
  }
}

static char *_atoa(char *str) {
  char *pbuf;

  pbuf = str;

  while(*pbuf != '\0') pbuf++;

  if(pbuf > str)
    pbuf--;

  while((pbuf > str) && (*pbuf == '\r' || *pbuf == '\n')) {
    *pbuf = '\0';
    pbuf--;
  }

  while((pbuf > str) && (*pbuf == ' ')) {
    *pbuf = '\0';
    pbuf--;
  }

  pbuf = str;
  while(*pbuf == '"' || *pbuf == ' ' || *pbuf == '\t') pbuf++;

  return pbuf;
}

static void client_open(session_t *session, session_commands_t *command, const char *cmd) {
  char  buf[MAX_LINE];
  char  *pbuf, *p;
  char  *host = NULL, *port = NULL;

  if((session == NULL) || (command == NULL))
    return;

  if(session->socket >= 0) {
    write_to_console(session, "Already connected to '%s:%s'.", session->host, session->port);
    return;
  }

  if(cmd) {
    strlcpy(buf, cmd, sizeof(buf));
    pbuf = buf;
    if((p = strchr(pbuf, ' ')) != NULL) {
      host = _atoa(p);
      if((port = strrchr(p, ':')) != NULL) {
	if(strlen(port) > 1) {
	  *port = '\0';
	  port++;
	}
	else {
	  *port = '\0';
	  port = NULL;
	}
      }
    }

    if(host != NULL) {

      strlcpy(session->host, host, sizeof(session->host));

      if(port != NULL)
	strlcpy(session->port, port, sizeof(session->port));

      session_create_commands(session);
      session->socket = sock_client(session->host, session->port, "tcp");
      if(session->socket < 0) {
	write_to_console(session, "opening server '%s' failed: %s.\nExiting.\n",
			 session->host, strerror(errno));

	session->running = 0;
      }
       _sock_write (session->socket, "commands\n", 9);
      session_update_prompt(session);
    }
    else {
      write_to_console(session, "open require arguments (host or host:port)\n");
    }
  }

}
static void client_quit (session_t *session, session_commands_t *command, const char *cmd) {
  client_close (session, command, cmd);
  pthread_mutex_lock (&session->console_mutex);
  session->running = 0;
  pthread_mutex_unlock (&session->console_mutex);
}

/*
 * End of client commands
 */

/*
 * completion generator functions.
 */
static char *command_generator(const char *text, int state) {
  static int   index, len;
  char        *cmd, *retcmd = NULL;

  if(!state) {
    index = 0;
    len = strlen(text);
  }

  if(len) {
    while((cmd = session_commands[index]->command) != NULL) {
      index++;
      if(session_commands[index - 1]->enable) {
	if(strncasecmp(cmd, text, len) == 0) {
	  retcmd = strdup(cmd);
	  return retcmd;
	}
      }
    }
  }

  return NULL;
}
static char **completion_function(const char *text, int start, int end) {
  char  **cmd = NULL;

  (void)end;
  if(start == 0)
    cmd = rl_completion_matches (text, command_generator);

  return cmd;
}

static void signals_handler (int sig) {
  switch(sig) {

    /* Kill the line on C-c */
  case SIGINT:
  case SIGTERM:
    if(rl_prompt) { /* readline is running, otherwise we are in script mode */
      rl_kill_full_line(1, 1);
      rl_redisplay();
    }
    else {
      if(session.socket >= 0) {
        _sock_write (session.socket, "exit\n", 5);
	close(session.socket);
      }
      exit(1);
    }
    break;

  }
}

static void *select_thread (void *data) {
  session_t       *session = (session_t *) data;
  fd_set           readfds;
  char             obuffer[MAX_LINE];
  int              ocount;
  char             buffer[MAX_LINE];
  int              size;
  int              i;
  char             c;
  int              was_down = 1;
  int              socket;
  int              running;

  FD_ZERO (&readfds);
  memset(&buffer, 0, sizeof(buffer));
  memset(&obuffer, 0, sizeof(obuffer));
  ocount = 0;

  while (1) {
    pthread_mutex_lock (&session->console_mutex);
    running = session->running;
    socket = session->socket;
    pthread_mutex_unlock (&session->console_mutex);
    if (!running)
      break;

    if (socket >= 0) {
      if (was_down) {
        struct timeval timeout = { 0, 200000 };
        FD_ZERO (&readfds);
        FD_SET (socket, &readfds);
        select (session->socket + 1, &readfds, (fd_set *) 0, (fd_set *) 0, &timeout);
	was_down = 0;
      }

      if(FD_ISSET(session->socket, &readfds)) {

        size = recvfrom (session->socket, buffer, sizeof(buffer), MSG_NOSIGNAL, NULL, NULL);

	if(size > 0) {

	  for(i = 0; i < size; i++) {
	    c = buffer[i];

	    switch (c) {
	    case '\r':
	      break;

	    case '\n':
	      obuffer[ocount++] = c;
	      if(i == (size - 1)) {
		int     pos = (strlen(session->prompt) + rl_end);
		char   *special;

		if((special = strstr(obuffer, COMMANDS_PREFIX)) != NULL) {
		  char  *p, *pp;
		  int    special_length = strlen(special);
		  size_t i = (sizeof(client_commands) / sizeof(client_commands[0])) - 1;

		  pp = special + 11;
		  while((p = xine_strsep(&pp, "\t")) != NULL) {
		    if(strlen(p)) {
		      while(*p == ' ' || *p == '\t') p++;
		      session_commands = (session_commands_t **) realloc(session_commands, (i + 2) * (sizeof(session_commands_t *)));
		      session_commands[i] = (session_commands_t *) malloc(sizeof(session_commands_t));
		      session_commands[i]->command  = strdup(p);
		      session_commands[i]->origin   = ORIGIN_SERVER;
		      session_commands[i]->enable   = 1;
		      session_commands[i]->function = client_noop;
		      i++;
		    }
		  }

		  /* remove '.\n' in last grabbed command */
		  session_commands[i - 1]->command[strlen(session_commands[i - 1]->command) - 1] = '\0';
		  session_commands[i - 1]->command[strlen(session_commands[i - 1]->command) - 1] = '\0';
		  session_commands[i] = (session_commands_t *) malloc(sizeof(session_commands_t));
		  session_commands[i]->command  = NULL;
		  session_commands[i]->origin   = ORIGIN_CLIENT;
		  session_commands[i]->enable   = 1;
		  session_commands[i]->function = NULL;

		  ocount -= special_length;
		  obuffer[ocount] = '\0';

		}
		pthread_mutex_lock(&session->console_mutex);
		/* Erase chars til's col0 */
		while(pos) {

		  write_to_console_unlocked_nocr(session, "\b \b");
		  pos--;
		}
		write_to_console_unlocked(session, "%s", obuffer);

		rl_crlf();
		rl_forced_update_display();

		pthread_mutex_unlock(&session->console_mutex);

		memset(&obuffer, 0, sizeof(obuffer));
		ocount = 0;
	      }
	      break;

	    default:
	      obuffer[ocount++] = c;
	      break;
	    }
	  }
	}
      }
    }
    else {
      was_down = 1;
    }
  }

  return NULL;
}

static void client_handle_command(session_t *session, const char *command) {
  int i, found, len1, len2;
  char cmd[MAX_LINE];

  if (!command)
    return;
  if (!command[0])
    return;

  /* Get only first arg of command */
  {
    const uint8_t *p = (const uint8_t *)command;
    while (tab_char_type[*p] & (2 + 4 + 8)) /* nl, ;, \t, spc */
      p++;
    len2 = strlcpy (cmd, (const char *)p, sizeof (cmd));
  }
  if (len2 > (int)sizeof (cmd) - 1)
    len2 = sizeof (cmd) - 1;
  len1 = len2;
  {
    char *p = strchr (cmd, ' ');
    if (p) {
      *p = 0;
      len1 = p - cmd;
    }
  }

  for (i = found = 0; (session_commands[i]->command && !found); i++) {
    if (session_commands[i]->enable) {
      if (!strncasecmp (cmd, session_commands[i]->command, len1)) {
        found++;
        if (session_commands[i]->origin == ORIGIN_CLIENT) {
          /* run locally */
          if (session_commands[i]->function)
            session_commands[i]->function (session, session_commands[i], command);
        } else {
          /* hand to server */
          cmd[len1] = ' ';
          cmd[len2] = '\n';
          if ((_sock_write (session->socket, cmd, len2 + 1)) == -1) {
            session->running = 0;
          }
        }
      }
    }
  }

  /* Perhaps a ';' separated commands, so send anyway to server */
  if(found == 0) {
    char buf[MAX_LINE], *p = buf, *e = buf + sizeof (buf) - 4;
    size_t l = strlen (command);
    if (p + l > e)
      l = e - p;
    memcpy (p, command, l); p += l;
    memcpy (p, "\n", 2); p += 1;
    _sock_write (session->socket, buf, p - buf);
  }

  if((!strncasecmp(cmd, "exit", strlen(cmd))) || (!strncasecmp(cmd, "halt", strlen(cmd)))) {
    session_create_commands(session);
    session->socket = -1;
  }

}

static void session_single_shot(session_t *session, int num_commands, char *commands[]) {
  int i;
  char buf[MAX_LINE];

  buf[0] = 0;

  for(i = 0; i < num_commands; i++) {
    if(buf[0])
      sprintf(buf+strlen(buf), " %s", commands[i]);
    else
      strcpy(buf, commands[i]);
  }

  client_handle_command(session, buf);
  usleep(10000);
  _sock_write (session->socket, "exit\n", 5);
}

static void show_version(void) {
  printf("This is %s - xine's remote control v%s.\n"
	 "(c) 2000-2019 The xine Team.\n", PROGNAME, PROGVERSION);
}

static void show_usage(void) {
  printf("usage: %s [options]\n", PROGNAME);
  printf("  -H, --host <hostname>        Connect host <hostname>.\n");
  printf("  -P, --port <port>            Connect using <port>.\n");
  printf("  -c, --command <command>      Send <command> to server then quit.\n");
  printf("  -n, --noconnect              Do not connect default server.\n");
  printf("  -v, --version                Display version.\n");
  printf("  -h, --help                   Display this help text.\n");
  printf("\n");
}

int main(int argc, char **argv) {
  int                c = '?';
  int                option_index = 0;
  char              *grabbed_line;
  struct sigaction   action;
  struct servent    *serv_ent;
  int                port_set = 0;
  int                auto_connect = 1;
  int                single_shot = 0;
  void              *p;

  /*
   * install sighandler.
   */
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGHUP) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGUSR1, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGUSR1) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGUSR2, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGUSR2) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGINT, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGINT) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGTERM, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGTERM) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGQUIT, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGQUIT) failed: %s\n", strerror(errno));
  }
  action.sa_handler = signals_handler;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;
  if(sigaction(SIGALRM, &action, NULL) != 0) {
    fprintf(stderr, "sigaction(SIGALRM) failed: %s\n", strerror(errno));
  }

  grabbed_line = NULL;

  session.socket = -1;
  pthread_mutex_init(&session.console_mutex, NULL);
  strcpy(session.host, DEFAULT_SERVER);
  strcpy(session.port, DEFAULT_XINECTL_PORT);

  opterr = 0;
  while((c = getopt_long(argc, argv, short_options, long_options, &option_index)) != EOF) {
    switch(c) {

    case 'H': /* Set host */
      if(optarg != NULL)
	strlcpy(session.host, optarg, sizeof(session.host));
      break;

    case 'P': /* Set port */
      if(optarg != NULL) {
	port_set = 1;
	strlcpy(session.port, optarg, sizeof(session.port));
      }
      break;

    case 'c': /* Execute argv[] command, then leave */
      single_shot = 1;
      break;

    case 'n': /* Disable automatic connection */
      auto_connect = 0;
      break;

    case 'v': /* Display version */
      show_version();
      exit(1);
      break;

    case 'h': /* Display usage */
    case '?':
      show_usage();
      exit(1);
      break;

    default:
      show_usage();
      fprintf(stderr, "invalid argument %d => exit\n", c);
      exit(1);
    }

  }

  if(single_shot)
    session.console = -1;
  else
    session.console = STDOUT_FILENO;

  /* Few realdline inits */
  rl_readline_name = PROGNAME;
  rl_set_prompt(session.prompt);
  rl_initialize();
  rl_attempted_completion_function = completion_function;

  signal(SIGPIPE, SIG_IGN);

  if(!port_set) {
    if((serv_ent = getservbyname("xinectl", "tcp")) != NULL) {
      snprintf(session.port, sizeof(session.port), "%u", ntohs(serv_ent->s_port));
    }
  }

  /* Prepare client commands */
  session_create_commands(&session);

  if(auto_connect) {
    session.socket = sock_client(session.host, session.port, "tcp");
    if(session.socket < 0) {
      fprintf(stderr, "opening server '%s' failed: %s\n", session.host, strerror(errno));
      exit(1);
    }
    /* Ask server for available commands */
    _sock_write (session.socket, "commands\n", 9);
  }

  write_to_console(&session, "? for help.\n");

  session.running = 1;
  pthread_create(&session.thread, NULL, select_thread, (void *)&session);

  if(single_shot) {
    session_single_shot(&session, (argc - optind), &argv[optind]);
    session.running = 0;
    goto __end;
  }

  while (session.running) {

    session_update_prompt(&session);

    if((grabbed_line = readline(session.prompt)) == NULL) {
      if (errno == 0 || errno == ENOTTY)
        exit(0);

      fprintf(stderr, "%s(%d): readline() failed: %s\n",
	      __XINE_FUNCTION__, __LINE__, strerror(errno));
      exit(1);
    }

    if (grabbed_line && grabbed_line[0]) {
      add_history (grabbed_line);
      client_handle_command (&session, grabbed_line);
    }
    SAFE_FREE(grabbed_line);
  }

 __end:

  if (session.socket >= 0) {
    close (session.socket);
    pthread_mutex_lock (&session.console_mutex);
    session.socket = -1;
    pthread_mutex_unlock (&session.console_mutex);
  }

  pthread_join (session.thread, &p);
  pthread_mutex_destroy (&session.console_mutex);
  printf ("\n");

  return 0;
}

/** /client only ********************************************************/

#else

/** server only *********************************************************/

#include <pwd.h>
#include <xine.h>
#include <xine/xineutils.h>

#include "panel.h"
#include "acontrol.h"
#include "playlist.h"
#include "mrl_browser.h"
#include "snapshot.h"
#include "control.h"
#include "actions.h"
#include "event.h"
#include "errors.h"

typedef struct commands_s commands_t;
typedef struct client_info_s client_info_t;
typedef struct passwds_s passwds_t;

/** return handled */
typedef int (*cmd_func_t) (client_info_t *info);

#define MAX_NAME_LEN    32
#define MAX_PASSWD_LEN  32

#define MAX_CLIENTS 4
struct xui_network_remote_s {
  gGui_t *gui;
  pthread_mutex_t mutex;
  int break_pipe[2];
  int fds[MAX_CLIENTS + 1];
  client_info_t *info[MAX_CLIENTS];
  passwds_t **passwds;
  int refs, current;
  pthread_t server;
};

struct commands_s {
#define CMD_ARGS_NONE 1
#define CMD_ARGS_OPTIONAL 0
#define CMD_ARGS_NEED 2
#define CMD_PUBLIC 4
#define CMD_NEED_AUTH 8
  uint32_t      flags;
  cmd_func_t    function;
  const char   *help;
  const char   *syntax;
};

typedef struct {
  uint8_t name[15], index;
} _name_list_t;

struct client_info_s {
  gGui_t               *gui;
  xui_network_remote_t *nr;
  int                   authentified;
  char                  name[MAX_NAME_LEN];
  char                  passwd[MAX_PASSWD_LEN];

  int                   index;
  int                   socket;
  union {
    struct sockaddr_in  in;
    struct sockaddr     sa;
  } fsin;

  int                  (*name_list_find) (const _name_list_t *list, uint32_t lsize, const char *word, uint32_t wsize);

  struct {
    uint16_t             num_args, buf_used;
    struct {
      uint16_t           start, len;
    }                    args[MAX_ARGS];
    char                 buf[MAX_LINE];
  } command;

};

/** Turn
 *  [\t do "exactly "what 'i say' then  return         \r\n]
 *  into
 *  [0] = [do]:2
 *  [1] = [exactly what]:12
 *  [2] = [i say]:5
 *  [3] = [then]:4
 *  [4] = [return]:6
 *  num_args = 5.
 *  Multiple commands are separated by newline and / or ";".
 */
static int _network_get_line (client_info_t *info, const char *line) {
  uint32_t n = 0;
  uint8_t flags;
  const uint8_t *p = (const uint8_t *)line;
  char *q = info->command.buf;

  /* skip separators and leading space */
  while (tab_char_type[*p] & (2 + 4 + 8)) /* nl, ;, \t, spc */
    p++;
  if (!*p)
    return 0;

  while (!(tab_char_type[*p] & (1 + 2)) && (n < MAX_ARGS)) { /* end, nl, ; */
    /* found start of word */
    info->command.args[n].start = q - info->command.buf;
    /* find end of word */
    while (1) {
      flags = tab_char_type[*p];
      /* plain char */
      if (!flags) {
        *q++ = *p++;
        continue;
      }
      if (flags & (1 + 2 + 4 + 8)) /* end, nl, ;, tab, spc */
        break;
      /* unquoting */
      if (flags & 16) { /* " */
        p++;
        while (!(flags = (tab_char_type[*p] & (1 + 2 + 16)))) { /* end, nl, ; " */
          while (!(flags & (1 + 2 + 16 + 64))) { /* end, nl, ;, ", \ */
            *q++ = *p++;
            flags = tab_char_type[*p];
          }
          if (flags & 64) { /* \ */
            flags = tab_char_type[*++p];
            if (flags & (1 + 2 + 16)) /* \0, nl, ;, " */
              break;
            *q++ = *p++;
            flags = tab_char_type[*p];
          }
        }
        if (flags & 16) /* " */
          p++;
      } else if (flags & 32) { /* ' */
        p++;
        while (!(flags = (tab_char_type[*p] & (1 + 2 + 32)))) { /* end, nl, ;, ' */
          while (!(flags & (1 + 2 + 32 + 64))) { /* end, nl, ;, ', \ */
            *q++ = *p++;
            flags = tab_char_type[*p];
          }
          if (flags & 64) { /* \ */
            flags = tab_char_type[*++p];
            if (flags & (1 + 2 + 32)) /* \0, nl, ;, ' */
              break;
            *q++ = *p++;
            flags = tab_char_type[*p];
          }
        }
        if (flags & 32) /* ' */
          p++;
      } else if (flags & 64) { /* \ */
        flags = tab_char_type[*++p];
        if (flags & (1 + 2)) /* end, nl, ; */
          break;
        *q++ = *p++;
      }
    }
    info->command.args[n].len = (q - info->command.buf) - info->command.args[n].start;
    *q++ = 0;
    n++;
    /* skip trailing space */
    while (tab_char_type[*p] & (4 + 8)) /* \t, spc */
      p++;
  }
  info->command.num_args = n;
  return p - (const uint8_t *)line;
}

/** The case insensitive and abbreviateable word parser. */
static int _name_list_find (const _name_list_t *list, uint32_t lsize, const char *word, uint32_t wsize) {
  uint8_t buf[sizeof (list[0].name) + 1];
  uint32_t i, f1, f2;
  if (!wsize)
    return -1;
  /* lowercase word */
  for (i = 0; i < sizeof (buf); i++)
    buf[i] = word[i] | 0x20;
  if (wsize > sizeof (buf))
    wsize = sizeof (buf);
  /* the char pos loop.
   * NOTE: gcc will unroll it because it runs at most 16 times.
   * Lets at least stop inlining the whole of _name_list_find () a dozen times
   * below by calling it through a pointer. saves ~18kbyte in executable file. */
  f1 = 0; f2 = lsize;
  for (i = 0; i < wsize; i++) {
    uint32_t b, l, m, e, found = 0;
    int d;
    /* find first match */
    b = f1; e = f2; m = (b + e) >> 1;
    do {
      d = (int)buf[i] - (int)list[m].name[i];
      found |= d == 0;
      if (d <= 0)
        e = m;
      else
        b = m + 1;
      l = m;
      m = (b + e) >> 1;
    } while (m != l);
    if (!found)
      return -1;
    f1 = e;
    /* find last match */
    b = f1; e = f2; m = (b + e) >> 1;
    do {
      d = (int)buf[i] - (int)list[m].name[i];
      if (d < 0)
        e = m;
      else
        b = m + 1;
      l = m;
      m = (b + e) >> 1;
    } while (m != l);
    f2 = e;
  }
  /* exact match? */
  if (list[f1].name[wsize] == 0)
    return f1;
  /* ambigiuity hit. return the lowest user index item. */
  {
    uint32_t li = f1, lu = list[f1].index;
    for (i = f1 + 1; i < f2; i++)
      if (list[i].index < lu)
        li = i, lu = list[i].index;
    return li;
  }
}

struct passwds_s {
  char ident[MAX_NAME_LEN];
  char passwd[MAX_PASSWD_LEN];
};

typedef struct {
  FILE             *fd;
  char             *ln;
  char              buf[256];
} fileobj_t;

enum {
  _I_none = 0,
  /* frequently used ones first, potentionally dangerous ones last. */
  _I_commands, _I_help, _I_syntax, _I_identify, _I_mrl, _I_play, _I_playlist, _I_stop,
  _I_pause, _I_exit, _I_fullscreen,
#ifdef HAVE_XINERAMA
  _I_xineramafull,
#endif
  _I_get, _I_set, _I_gui, _I_event, _I_seek, _I_halt, _I_snapshot,
  _I_last
};

static const _name_list_t cmd_names[] = {
  { "commands",         _I_commands },
  { "event",            _I_event },
  { "exit",             _I_exit },
  { "fullscreen",       _I_fullscreen },
  { "get",              _I_get },
  { "gui",              _I_gui },
  { "halt",             _I_halt },
  { "help",             _I_help },
  { "identify",         _I_identify },
  { "mrl",              _I_mrl },
  { "pause",            _I_pause },
  { "play",             _I_play },
  { "playlist",         _I_playlist },
  { "seek",             _I_seek },
  { "set",              _I_set },
  { "snapshot",         _I_snapshot },
  { "stop",             _I_stop },
  { "syntax",           _I_syntax },
#ifdef HAVE_XINERAMA
  { "xineramafull",     _I_xineramafull }
#endif
};

static const commands_t commands[_I_last];

static int sock_serv(const char *service, const char *transport, int queue_length) {
  union {
    struct sockaddr_in in;
    struct sockaddr sa;
  } fsin;
  int                 sock;
  int                 on = 1;

  sock = sock_create(service, transport, &fsin.in);
  if (sock < 0) {
    return -1;
  }

  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(int));

  if(bind(sock, &fsin.sa, sizeof(fsin.in)) < 0)
    fprintf (stderr, "Unable to link socket %s: %s\n", service, strerror(errno));

  if(strcmp(transport, "udp") && listen(sock, queue_length) < 0)
    fprintf (stderr, "Passive mode impossible on %s: %s\n", service, strerror(errno));

  return sock;
}

/*
 * Read from socket.
 */
static ssize_t sock_read(int socket, char *buf, size_t len) {
  char    *pbuf;
  ssize_t  r, rr;
  void    *nl;

  if((socket < 0) || (buf == NULL) || (len < 1))
    return -1;

  if(!sock_check_opened(socket))
    return -1;

  if (--len < 1)
    return(-1);

  pbuf = buf;

  do {

    if((r = recv(socket, pbuf, len, MSG_PEEK)) <= 0)
      return -1;

    if((nl = memchr(pbuf, '\n', r)) != NULL)
      r = ((char *) nl) - pbuf + 1;

    if((rr = read(socket, pbuf, r)) < 0)
      return -1;

    pbuf += rr;
    len -= rr;

  } while((nl == NULL) && len);

  *pbuf = '\0';

  return (pbuf - buf);
}

/*
 * Password related
 */
static void _passwdfile_get_next_line(fileobj_t *fobj) {
  char *p;

 __get_next_line:

  fobj->ln = fgets(fobj->buf, sizeof(fobj->buf), fobj->fd);

  while(fobj->ln && (*fobj->ln == ' ' || *fobj->ln == '\t')) ++fobj->ln;

  if(fobj->ln) {
    if((strncmp(fobj->ln, ";", 1) == 0) ||
       (strncmp(fobj->ln, "#", 1) == 0) ||
       (*fobj->ln == '\0')) {
      goto __get_next_line;
    }

  }

  p = fobj->ln;

  if(p) {
    while(*p != '\0') {
      if(*p == '\n' || *p == '\r') {
	*p = '\0';
	break;
      }
      p++;
    }

    while(p > fobj->ln) {
      --p;

      if(*p == ' ' || *p == '\t')
	*p = '\0';
      else
	break;
    }
  }

}
static int _passwdfile_is_entry_valid (char *entry, passwds_t *pass) {
  char buf[MAX_LINE];
  char *n, *p;

  strlcpy(buf, entry, sizeof(buf));
  n = buf;
  if((p = strrchr(buf, ':')) != NULL) {
    if(strlen(p) > 1) {
      *p = '\0';
      p++;
    }
    else {
      *p = '\0';
      p = NULL;
    }
  }

  if((n != NULL) && (p != NULL)) {
    strlcpy (pass->ident, n, MAX_NAME_LEN);
    strlcpy (pass->passwd, p, MAX_PASSWD_LEN);
    return 1;
  }

  return 0;
}

static int server_load_passwd (xui_network_remote_t *nr, const char *passwdfilename) {
  fileobj_t   fobj;
  passwds_t  *pass;
  int         entries = 0;

  fobj.fd = fopen (passwdfilename, "r");
  if (!fobj.fd) {
    fprintf (stderr, "fopen() failed: %s\n", strerror (errno));
    return 0;
  }

  pass = (passwds_t *)malloc (sizeof *pass);
  if (!pass) {
    fclose (fobj.fd);
    return 0;
  }

  _passwdfile_get_next_line (&fobj);
  while (fobj.ln != NULL) {
    if (_passwdfile_is_entry_valid (fobj.buf, pass)) {
      passwds_t **np = (passwds_t **)realloc (nr->passwds, sizeof (passwds_t *) * (entries + 2));
      if (!np)
        break;
      nr->passwds = np;
      np[entries] = pass;
      pass = (passwds_t *)malloc (sizeof *pass);
      if (!pass)
        break;
      entries++;
    }
    _passwdfile_get_next_line (&fobj);
  }
  nr->passwds[entries] = NULL;
  free (pass);
  fclose (fobj.fd);
  return entries;
}

static int is_client_authorized (client_info_t *info) {
  xui_network_remote_t *nr = info->nr;
  int all = 0, user = 0, i;

  for (i = 0; nr->passwds[i]; i++) {
    if (!strcmp (nr->passwds[i]->ident, "ALL")) {
      if (!strcmp (nr->passwds[i]->passwd, "ALLOW"))
        all = 1;
      else if (!strcmp (nr->passwds[i]->passwd, "DENY"))
        all = 2;
      else
        all = 0;
    } else if (!strcasecmp (nr->passwds[i]->ident, info->name)) {
      if (nr->passwds[i]->passwd[0] == '*') /* user blocked regardless of password. */
        return 0;
      if (!strcmp (nr->passwds[i]->passwd, info->passwd)) /* password OK */
        user = 1;
    }
  }
  return user || (all == 1);
}

/*
 * Check access rights.
 */
static void check_client_auth (client_info_t *info) {
  char buf[MAX_LINE], *p = buf, *e = buf + sizeof (buf) - 4;

  if (is_client_authorized (info)) {
    info->authentified = 1;
    p += snprintf (p, e - p, "user '%s' has been authentified.\n", info->name);
  } else {
    info->authentified = 0;
    p += snprintf (p, e - p, "user '%s' isn't known/authorized.\n", info->name);
  }
  _sock_write (info->socket, buf, p - buf);
}

static void network_err_msg (gGui_t *gui, const char *text) {
  xui_network_remote_t *nr = gui ? gui->network_remote : NULL;

  if (nr)
    _sock_write (nr->fds[nr->current], text, strlen (text));
}

/*
 * ************* COMMANDS ************
 */
static int do_commands (client_info_t *client_info) {
  uint32_t i;
  char buf[MAX_LINE], *p = buf;

  strcpy (p, COMMANDS_PREFIX); p += strlen (COMMANDS_PREFIX);

  for (i = 0; i < sizeof (cmd_names) / sizeof (cmd_names[0]); i++) {
    if (!(commands[cmd_names[i].index].flags & CMD_PUBLIC))
      continue;
    *p++ = '\t';
    strcpy (p, (const char *)cmd_names[i].name); p += strlen ((const char *)cmd_names[i].name);
  }
  memcpy (p, ".\n\n", 4); p += 3;
  _sock_write (client_info->socket, buf, p - buf);
  return 1;
}

static size_t _list_cmds (char *buf, size_t bsize, int line_width) {
  char *p = buf, *s = buf, *e = buf + bsize - 1;
  int cwidth = 0, cnum, col = 0;
  uint32_t i;

  for (i = 0; i < sizeof (cmd_names) / sizeof (cmd_names[0]); i++) {
    int l;
    if (!(commands[cmd_names[i].index].flags & CMD_PUBLIC))
      continue;
    l = strlen ((const char *)cmd_names[i].name);
    if (l > cwidth)
      cwidth = l;
  }
  cwidth += 1;
  cnum = (line_width - 8) / cwidth;
  if (cnum <= 0)
    cnum = 1;

  for (i = 0; i < sizeof (cmd_names) / sizeof (cmd_names[0]); i++) {
    int l;
    if (!(commands[cmd_names[i].index].flags & CMD_PUBLIC))
      continue;
    l = strlen ((const char *)cmd_names[i].name);

    if (col == 0) {
      if (p + 8 > e)
        break;
      memset (p, ' ', 8); p += 8;
    }
    if (p + cwidth > e)
      break;
    memcpy (p, cmd_names[i].name, l); p += l;
    s = p;
    if (++col < cnum) {
      memset (p, ' ', cwidth - l); p += cwidth - l;
    } else {
      if (p >= e)
        break;
      *p++ = '\n';
      col = 0;
    }
  }
  p = s;
  if (p < e)
    *p++ = '\n';
  *p = 0;
  return p - buf;
}

static int do_help (client_info_t *info) {
  char buf[MAX_LINE], *p = buf, *e = buf + sizeof (buf) - 4;

  if (info->command.num_args < 2) {
    p += strlcpy (p, "Available commands are:\n", e - p);
    p += _list_cmds (p, e - p, 80);
  } else {
    const char *name = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (cmd_names, sizeof (cmd_names) / sizeof (cmd_names[0]), name, info->command.args[1].len);
    if (i >= 0) {
      p += snprintf (p, e - p, "Help of '%s' command:\n  %s\n%s",
        (const char *)cmd_names[i].name, commands[cmd_names[i].index].help, commands[cmd_names[i].index].syntax);
    } else {
      p += snprintf (p, e - p, "Unknown command '%s'.\n", name);
    }
  }
  _sock_write (info->socket, buf, p - buf);
  return 1;
}

static int do_syntax (client_info_t *info) {
  char buf[MAX_LINE], *p = buf, *e = buf + sizeof (buf);

  if (info->command.num_args >= 2) {
    const char *name = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (cmd_names, sizeof (cmd_names) / sizeof (cmd_names[0]), name, info->command.args[1].len);
    if (i >= 0) {
       p += snprintf (p, e - p, "Syntax of '%s' command:\n%s\n",
        (const char *)cmd_names[i].name, commands[cmd_names[i].index].syntax);
    } else {
      p += snprintf (p, e - p, "Unknown command '%s'.\n\n", name);
    }
    _sock_write (info->socket, buf, p - buf);
  }
  return 1;
}

static int do_auth (client_info_t *info) {
  int nargs = info->command.num_args - 1;

  if (nargs > 0) {
    if (nargs >= 1) {
      char buf[MAX_LINE];
      char *name;
      char *passwd;

      strlcpy (buf, info->command.buf + info->command.args[1].start, sizeof (buf));
      name = buf;
      if((passwd = strrchr(buf, ':')) != NULL) {
	if(strlen(passwd) > 1) {
	  *passwd = '\0';
	  passwd++;
	}
	else {
	  *passwd = '\0';
	  passwd = NULL;
	}
      }
      if (name && passwd) {
        memset (&info->name, 0, sizeof (info->name));
        memset (&info->passwd, 0, sizeof (info->passwd));
        strlcpy (info->name, name, sizeof (info->name));
        strlcpy (info->passwd, passwd, sizeof (info->passwd));
        check_client_auth (info);
      } else {
        _sock_write (info->socket, "use identity:password syntax.\n", 30);
      }
    }
    return 1;
  }
  return 0;
}

static int do_play (client_info_t *info) {
  gGui_t *gui = info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_execute_action_id (gui, ACTID_PLAY);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

static int do_mrl (client_info_t *info) {
  gGui_t *gui = info->gui;
  int res = 0;

  if (info->command.num_args >= 2) {
    static const _name_list_t words[] = {
      { "add",  0 },
      { "next", 1 },
      { "play", 2 },
      { "prev", 3 }
    };
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (words, sizeof (words) / sizeof (words[0]), arg, info->command.args[1].len);

    switch (i) {

      case 0: /* "add" */
        {
          uint32_t i;
          xitk_lock (gui->xitk, 1);
          gui->nongui_error_msg = network_err_msg;
          for (i = 2; i < info->command.num_args; i++)
            gui_dndcallback (gui, info->command.buf + info->command.args[i].start);
          gui->nongui_error_msg = NULL;
          xitk_lock (gui->xitk, 0);
        }
        res = 1;
        break;

      case 1: /* "next" */
        i = 1;
        goto _snext;

      case 2: /* "play" */
        if (info->command.num_args >= 3) {
          xitk_lock (gui->xitk, 1);
          gui->nongui_error_msg = network_err_msg;
          gui_dndcallback (gui, info->command.buf + info->command.args[2].start);
          gui_playlist_play (gui, GUI_MMK_LAST);
          gui->nongui_error_msg = NULL;
          xitk_lock (gui->xitk, 0);
          res = 1;
        }
        break;

      case 3: /* "prev" */
        i = -1;
      _snext:
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_start_next (gui, i);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      default: ;
    }
  }
  return res;
}

static int do_playlist (client_info_t *info) {
  gGui_t *gui = info->gui;
  int res = 0;

  if (info->command.num_args >= 2) {
    static const _name_list_t words[] = {
      { "continue", 0 },
      { "delete",   1 },
      { "first",    2 },
      { "last",     3 },
      { "next",     4 },
      { "prev",     5 },
      { "select",   6 },
      { "show",     7 },
      { "stop",     8 }
    };
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (words, sizeof (words) / sizeof (words[0]), arg, info->command.args[1].len);

    switch (i) {

      case 0: /* "continue" */
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_lock (gui);
        if (xine_get_status(gui->stream) != XINE_STATUS_STOP)
          gui->playlist.control &= ~PLAYLIST_CONTROL_STOP;
        gui_playlist_unlock (gui);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      case 1: /* "delete" */
        if (info->command.num_args >= 3) {
          arg = info->command.buf + info->command.args[2].start;
          if (!strcmp (arg, "all") || !strcmp (arg, "*")) {
            xitk_lock (gui->xitk, 1);
            gui->nongui_error_msg = network_err_msg;
            gui_playlist_free (gui);
            if (xine_get_status(gui->stream) != XINE_STATUS_STOP) {
              gui_stop (NULL, gui);
              panel_playback_ctrl (gui->panel, 0);
            }
            gui_playlist_remove (gui, 0, GUI_PLAYLIST_REMOVE_ALL);
          } else {
            int j = atoi (arg);
            xitk_lock (gui->xitk, 1);
            gui->nongui_error_msg = network_err_msg;
            if (j >= 0) {
              if ((gui->playlist.cur == j) && ((xine_get_status (gui->stream) != XINE_STATUS_STOP)))
                gui_stop (NULL, gui);
              gui_playlist_remove (gui, j, GUI_PLAYLIST_REMOVE_1);
            }
          }
          playlist_update_playlist (gui);
          if (gui->playlist.num > 0) {
            gui_current_set_index (gui, GUI_MMK_CURRENT);
          } else {
            panel_playback_ctrl (gui->panel, 0);
            if (xine_get_status (gui->stream) != XINE_STATUS_STOP)
              gui_stop (NULL, gui);
            gui_current_set_index (gui, GUI_MMK_NONE);
          }
          gui->nongui_error_msg = NULL;
          xitk_lock (gui->xitk, 0);
          res = 1;
        }
        break;

      case 2: /* "first" */
        i = 0;
        goto _select;

      case 3: /* "last" */
        i = GUI_MMK_LAST;
        goto _select;

      case 4: /* "next", Alias of mrl next */
        i = 1;
        goto _snext;

      case 5: /* "prev", Alias of mrl prev */
        i = -1;
      _snext:
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_start_next (gui, i);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      case 6: /* "select" */
        if (info->command.num_args < 3)
          break;
        i = atoi (info->command.buf + info->command.args[2].start);
      _select:
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_play (gui, i);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      case 7: /* "show" */
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_lock (gui);
        if (gui->playlist.num) {
          int i;
          char buf[MAX_LINE], *p, *e = buf + sizeof (buf) - 4;
          for (i = 0; i < gui->playlist.num; i++) {
            p = buf;
            p += snprintf (p, e - p, "%2s %5d %s\n",
              (i == gui->playlist.cur) ? "*>" : "", i, gui->playlist.mmk[i]->mrl);
            _sock_write (info->socket, buf, p - buf);
          }
        } else {
          _sock_write (info->socket, "empty playlist.\n", 16);
        }
        gui_playlist_unlock (gui);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      case 8: /* "stop" */
        xitk_lock (gui->xitk, 1);
        gui->nongui_error_msg = network_err_msg;
        gui_playlist_lock (gui);
        if (xine_get_status (gui->stream) != XINE_STATUS_STOP)
          gui->playlist.control |= PLAYLIST_CONTROL_STOP;
        gui_playlist_unlock (gui);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        res = 1;
        break;

      default: ;
    }
  }
  return res;
}

static int do_stop (client_info_t *info) {
  gGui_t *gui = info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_execute_action_id (gui, ACTID_STOP);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

static int do_pause (client_info_t *info) {
  gGui_t *gui = info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_execute_action_id (gui, ACTID_PAUSE);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

static int do_fullscreen (client_info_t *info) {
  gGui_t *gui = info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_execute_action_id (gui, ACTID_TOGGLE_FULLSCREEN);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

#ifdef HAVE_XINERAMA
static int do_xinerama_fullscreen (client_info_t *info) {
  gGui_t *gui = info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_execute_action_id (gui, ACTID_TOGGLE_XINERAMA_FULLSCREEN);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}
#endif

static const _name_list_t _words1[] = {
  { "audio",    0 },
  { "av",       1 },
  { "length",   2 },
  { "loop",     3 },
  { "position", 4 },
  { "speed",    5 },
  { "spu",      6 },
  { "status",   7 }
};
static const _name_list_t _words2[] = {
  { "channel", 0 },
  { "lang",    1 },
  { "mute",    2 },
  { "offset",  3 },
  { "volume",  4 }
};

static int do_get (client_info_t *info) {
  gGui_t *gui = info->gui;
  char buf[256], *p = buf, *e = buf + sizeof (buf);
  int res = 0;

  if (info->command.num_args >= 2) {
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (_words1, sizeof (_words1) / sizeof (_words1[0]), arg, info->command.args[1].len);

    switch (i) {

      case 0: /* "audio" */
        if (info->command.num_args >= 3) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
            case 0: /* "channel" */
              xitk_lock (gui->xitk, 1);
              i = xine_get_param (gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL);
              xitk_lock (gui->xitk, 0);
              p += snprintf (p, e - p, "Current audio channel: %d\n\n", i);
              res = 1;
              break;
            case 1: /* "lang" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              {
                char lbuf[XINE_LANG_MAX];
                int channel = xine_get_param(gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL);
                lbuf[0] = 0;
                if (!xine_get_audio_lang (gui->stream, channel, lbuf))
                  snprintf (lbuf, sizeof (lbuf), "%3d", channel);
                p += snprintf (p, e - p, "Current audio language: %s\n\n", lbuf);
              }
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            case 2: /* "mute" */
              xitk_lock (gui->xitk, 1);
              i = gui->mixer.mute[gui->mixer.type_mute];
              xitk_lock (gui->xitk, 0);
              p += snprintf (p, e - p, "Current audio mute: %d\n\n", i);
              res = 1;
              break;
            case 4: /* "volume" */
              xitk_lock (gui->xitk, 1);
              i = gui->mixer.level[gui->mixer.type_volume];
              xitk_lock (gui->xitk, 0);
              if (i >= 0) {
                p += snprintf (p, e - p, "Current audio volume: %d\n\n", i);
              } else {
                memcpy (p, "Audio is disabled.\n\n", 20); p += 20;
              }
              res = 1;
              break;
            default: ;
          }
        }
        break;

      case 1: /* "av" */
        if (info->command.num_args >= 3) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
            case 3: /* "offset" */
              {
                xitk_lock (gui->xitk, 1);
                i = xine_get_param(gui->stream, XINE_PARAM_AV_OFFSET);
                xitk_lock (gui->xitk, 0);
                p += snprintf (p, e - p, "Current A/V offset: %d\n\n", i);
              }
              res = 1;
              break;
            default: ;
          }
        }
        break;

      case 2: /* "length" */
        {
          int pos_stream, pos_time, length_time;
          xitk_lock (gui->xitk, 1);
          xine_get_pos_length (gui->stream, &pos_stream, &pos_time, &length_time);
          xitk_lock (gui->xitk, 0);
          p += snprintf (p, e - p, "Current length: %d\n\n", length_time);
        }
        res = 1;
        break;

      case 3: /* "loop" */
        {
          int mode;
          xitk_lock (gui->xitk, 1);
          mode = gui->playlist.loop;
          xitk_lock (gui->xitk, 0);
          memcpy (p, "Current loop mode is: ", 22); p += 22;
          switch (mode) {
            case PLAYLIST_LOOP_NO_LOOP:
              memcpy (p, "'No Loop'", 9); p += 9;
              break;
            case PLAYLIST_LOOP_LOOP:
              memcpy (p, "'Loop'", 6); p += 6;
              break;
            case PLAYLIST_LOOP_REPEAT:
              memcpy (p, "'Repeat'", 8); p += 8;
              break;
            case PLAYLIST_LOOP_SHUFFLE:
              memcpy (p, "'Shuffle'", 9); p += 9;
              break;
            case PLAYLIST_LOOP_SHUF_PLUS:
              memcpy (p, "'Shuffle forever'", 17); p += 17;
              break;
            default:
              memcpy (p, "'!!Unknown!!'", 13); p += 13;
          }
          memcpy (p, ".\n\n", 4); p += 3;
        }
        res = 1;
        break;

      case 4: /* "position" */
        {
          int pos_stream, pos_time, length_time;
          xitk_lock (gui->xitk, 1);
          xine_get_pos_length (gui->stream, &pos_stream, &pos_time, &length_time);
          xitk_lock (gui->xitk, 0);
          p += snprintf (p, e - p, "Current position: %d\n\n", pos_time);
        }
        res = 1;
        break;

      case 5: /* "speed" */
        {
          static const uint8_t map[] = { 0, 1, 2,2, 3, 4,4,4,4,4,4,4,5,5,5,5,5, 6 };
          static const char names[][12] = {
            "PAUSE", "SLOW_4", "SLOW_2", "NORMAL", "FAST_2", "FAST_4", "UNKNOWN"
          };
          int speed;
          xitk_lock (gui->xitk, 1);
          speed = xine_get_param (gui->stream, XINE_PARAM_SPEED);
          xitk_lock (gui->xitk, 0);
          memcpy (p, "Current speed: ", 15); p += 15;
          if ((unsigned int)speed < 17) {
            memcpy (p, "XINE_SPEED_", 12); p += 11;
          } else {
            speed = 17;
          }
          p += strlcpy (p, names[map[speed]], e - p);
          memcpy (p, "\n\n", 3); p += 2;
        }
        res = 1;
        break;

      case 6: /* "spu" */
        if (info->command.num_args >= 3) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
            case 0: /* "channel" */
              xitk_lock (gui->xitk, 1);
              i = xine_get_param (gui->stream, XINE_PARAM_SPU_CHANNEL);
              xitk_lock (gui->xitk, 0);
              p += snprintf (p, e - p, "Current spu channel: %d\n\n", i);
              res = 1;
              break;
            case 1: /* "lang" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              {
                char lbuf[XINE_LANG_MAX];
                int channel = xine_get_param (gui->stream, XINE_PARAM_SPU_CHANNEL);
                lbuf[0] = 0;
                if (!xine_get_spu_lang (gui->stream, channel, &buf[0]))
                  snprintf (buf, sizeof(buf), "%3d", channel);
                p += snprintf (p, e - p, "Current spu language: %s\n\n", lbuf);
              }
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            case 3: /* "offset" */
              xitk_lock (gui->xitk, 1);
              i = xine_get_param (gui->stream, XINE_PARAM_SPU_OFFSET);
              xitk_lock (gui->xitk, 0);
              p += snprintf (p, e - p, "Current spu offset: %d\n\n", i);
              res = 1;
              break;
            default: ;
          }
        }
        break;

      case 7: /* "status" */
        {
          static const char names[XINE_STATUS_QUIT + 1][20] = {
            [XINE_STATUS_IDLE] = "XINE_STATUS_IDLE",
            [XINE_STATUS_PLAY] = "XINE_STATUS_PLAY",
            [XINE_STATUS_STOP] = "XINE_STATUS_STOP",
            [XINE_STATUS_QUIT] = "XINE_STATUS_QUIT"
          };
          int status;
          xitk_lock (gui->xitk, 1);
          status = xine_get_status (gui->stream);
          xitk_lock (gui->xitk, 0);
          memcpy (buf, "Current status: ", 16); p += 16;
          p += strlcpy (p, (status <= XINE_STATUS_QUIT) ? names[status] : "*UNKNOWN*", e - p);
          memcpy (p, "\n\n", 3); p += 2;
        }
        res = 1;
        break;

      default: ;
    }
  }
  if (p > buf)
    _sock_write (info->socket, buf, p - buf);
  return res;
}

static int do_set (client_info_t *info) {
  gGui_t *gui = info->gui;
  int res = 0;

  if (info->command.num_args >= 3) {
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (_words1, sizeof (_words1) / sizeof (_words1[0]), arg, info->command.args[1].len);

    switch (i) {

      case 0: /* "audio" */
        if (info->command.num_args >= 4) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
            case 0: /* "channel" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              xine_set_param (gui->stream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL,
                atoi (info->command.buf + info->command.args[3].start));
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            case 2: /* "mute" */
              {
                static const int _param[LAST_MIXER] = {
                  [SOUND_CARD_MIXER] = XINE_PARAM_AUDIO_MUTE,
                  [SOFTWARE_MIXER] = XINE_PARAM_AUDIO_AMP_MUTE
                };
                xitk_lock (gui->xitk, 1);
                gui->nongui_error_msg = network_err_msg;
                gui->mixer.mute[gui->mixer.type_mute] = xitk_get_bool_value (info->command.buf + info->command.args[3].start);
                xine_set_param (gui->stream, _param[gui->mixer.type_mute], gui->mixer.mute[gui->mixer.type_mute]);
                gui->nongui_error_msg = NULL;
                xitk_lock (gui->xitk, 0);
              }
              res = 1;
              break;
            case 4: /* "volume" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              if (gui->mixer.level[gui->mixer.type_volume] >= 0) {
                static const uint8_t _max[LAST_MIXER] = {
                  [SOUND_CARD_MIXER] = 100, [SOFTWARE_MIXER] = 200
                };
                static const int _param[LAST_MIXER] = {
                  [SOUND_CARD_MIXER] = XINE_PARAM_AUDIO_VOLUME, [SOFTWARE_MIXER] = XINE_PARAM_AUDIO_AMP_LEVEL
                };
                int vol = atoi (info->command.buf + info->command.args[3].start);
                if (vol < 0)
                  vol = 0;
                else if (vol > (int)_max[gui->mixer.type_volume])
                  vol = _max[gui->mixer.type_volume];
                gui->mixer.level[gui->mixer.type_volume] = vol;
                xine_set_param(gui->stream, _param[gui->mixer.type_volume], gui->mixer.level[gui->mixer.type_volume]);
              } else {
                _sock_write (info->socket, "Audio is disabled.\n", 19);
              }
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            default: ;
          }
        }
        break;

      case 1: /* "av" */
        if (info->command.num_args >= 4) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
	    case 3: /* "offset" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              xine_set_param (gui->stream, XINE_PARAM_AV_OFFSET, atoi (info->command.buf + info->command.args[3].start));
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            default: ;
          }
        }
        break;

      case 3: /* "loop" */
        {
          static const _name_list_t _words3[] = {
            { "loop",     PLAYLIST_LOOP_LOOP      },
            { "no",       PLAYLIST_LOOP_NO_LOOP   },
            { "repeat",   PLAYLIST_LOOP_REPEAT    },
            { "shuffle",  PLAYLIST_LOOP_SHUFFLE   },
            { "shuffle+", PLAYLIST_LOOP_SHUF_PLUS }
          };
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words3, sizeof (_words3) / sizeof (_words3[0]), arg, info->command.args[2].len);
          if (i >= 0) {
            xitk_lock (gui->xitk, 1);
            gui->playlist.loop = _words3[i].index;
            xitk_lock (gui->xitk, 0);
          }
        }
        res = 1;
        break;

      case 5: /* "speed" */
        {
          int speed = -1;
          arg = info->command.buf + info->command.args[2].start;
          if (!memcmp (arg, "XINE_SPEED_", 11)) {
            arg += 11;
            if (!memcmp (arg, "PAUSE", 5))
              speed = XINE_SPEED_PAUSE;
            else if (!memcmp (arg, "SLOW_4", 6))
              speed = XINE_SPEED_SLOW_4;
            else if (!memcmp (arg, "SLOW_2", 6))
              speed = XINE_SPEED_SLOW_2;
            else if (!memcmp (arg, "NORMAL", 6))
              speed = XINE_SPEED_NORMAL;
            else if (!memcmp (arg, "FAST_2", 6))
              speed = XINE_SPEED_FAST_2;
            else if (!memcmp (arg, "FAST_4", 6))
              speed = XINE_SPEED_FAST_4;
          } else {
            if (!memcmp (arg, "|", 1))
              speed = XINE_SPEED_PAUSE;
            else if (!memcmp (arg, "/4", 2))
              speed = XINE_SPEED_SLOW_4;
            else if (!memcmp (arg, "/2", 2))
              speed = XINE_SPEED_SLOW_2;
            else if (!memcmp (arg, "=", 1))
              speed = XINE_SPEED_NORMAL;
            else if (!memcmp (arg, "*2", 2))
              speed = XINE_SPEED_FAST_2;
            else if (!memcmp (arg, "*4", 2))
              speed = XINE_SPEED_FAST_4;
          }
          if (speed < 0)
            speed = atoi (arg);
          xitk_lock (gui->xitk, 1);
          gui->nongui_error_msg = network_err_msg;
          xine_set_param(gui->stream, XINE_PARAM_SPEED, speed);
          gui->nongui_error_msg = NULL;
          xitk_lock (gui->xitk, 0);
        }
        res = 1;
        break;

      case 6: /* "spu" */
        if (info->command.num_args >= 4) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (_words2, sizeof (_words2) / sizeof (_words2[0]), arg, info->command.args[2].len);
          switch (i) {
            case 0: /* "channel" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              xine_set_param (gui->stream, XINE_PARAM_SPU_CHANNEL, atoi (info->command.buf + info->command.args[3].start));
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            case 3: /* "offset" */
              xitk_lock (gui->xitk, 1);
              gui->nongui_error_msg = network_err_msg;
              xine_set_param (gui->stream, XINE_PARAM_SPU_OFFSET, atoi (info->command.buf + info->command.args[3].start));
              gui->nongui_error_msg = NULL;
              xitk_lock (gui->xitk, 0);
              res = 1;
              break;
            default: ;
          }
        }
        break;

      default: ;
    }
  }
  return res;
}

static int do_gui (client_info_t *info) {
  gGui_t *gui = info->gui;

  if (info->command.num_args >= 2) {
    static const _name_list_t words1[] = {
      { "acontrol", ACTID_ACONTROLSHOW },
      { "apost",    ACTID_APP },
      { "control",  ACTID_CONTROLSHOW },
      { "help",     ACTID_HELP_SHOW },
      { "hide",     ACTID_TOGGLE_VISIBLITY },
      { "log",      ACTID_VIEWLOG },
      { "mrl",      ACTID_MRLBROWSER },
      { "output",   ACTID_TOGGLE_WINOUT_VISIBLITY },
      { "panel",    ACTID_TOGGLE_VISIBLITY },
      { "playlist", ACTID_PLAYLIST },
      { "setup",    ACTID_SETUP },
      { "vpost",    ACTID_VPP }
    };
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (words1, sizeof (words1) / sizeof (words1[0]), arg, info->command.args[1].len);

    if (i < 0)
      return 0;
    xitk_lock (gui->xitk, 1);
    gui->nongui_error_msg = network_err_msg;
    gui_execute_action_id (gui, words1[i].index);
    gui->nongui_error_msg = NULL;
    xitk_lock (gui->xitk, 0);
    return 1;
  }
  return 0;
}

static int do_event (client_info_t *info) {
  gGui_t *gui = info->gui;
  xine_event_t xine_event;

  xine_event.type = 0;
  if (info->command.num_args >= 2) {
    static const _name_list_t words[] = {
      { "+10",      XINE_EVENT_INPUT_NUMBER_10_ADD },
      { "angle",    XINE_EVENT_INPUT_ANGLE_NEXT },
      { "down",     XINE_EVENT_INPUT_DOWN },
      { "left",     XINE_EVENT_INPUT_LEFT },
      { "menu1",    XINE_EVENT_INPUT_MENU1 },
      { "menu2",    XINE_EVENT_INPUT_MENU2 },
      { "menu3",    XINE_EVENT_INPUT_MENU3 },
      { "menu4",    XINE_EVENT_INPUT_MENU4 },
      { "menu5",    XINE_EVENT_INPUT_MENU5 },
      { "menu6",    XINE_EVENT_INPUT_MENU6 },
      { "menu7",    XINE_EVENT_INPUT_MENU7 },
      { "next",     XINE_EVENT_INPUT_NEXT },
      { "previous", XINE_EVENT_INPUT_PREVIOUS },
      { "right",    XINE_EVENT_INPUT_RIGHT },
      { "select",   XINE_EVENT_INPUT_SELECT },
      { "up",       XINE_EVENT_INPUT_UP }
    };
    const char *arg = info->command.buf + info->command.args[1].start;
    int i = info->name_list_find (words, sizeof (words) / sizeof (words[0]), arg, info->command.args[1].len);
      if (i >= 0) {
      if (words[i].index == XINE_EVENT_INPUT_ANGLE_NEXT) {
        if (info->command.num_args >= 3) {
          arg = info->command.buf + info->command.args[2].start;
          i = info->name_list_find (words, sizeof (words) / sizeof (words[0]), arg, info->command.args[2].len);
          xine_event.type = (i == 11) ? XINE_EVENT_INPUT_ANGLE_NEXT
                          : (i == 12) ? XINE_EVENT_INPUT_ANGLE_PREVIOUS
                          : 0;
        }
      } else {
        xine_event.type = words[i].index;
      }
    } else {
      const uint8_t *d = (const uint8_t *)arg;
      uint8_t z = *d ^ '0';
      if (z < 10u)
        xine_event.type = XINE_EVENT_INPUT_NUMBER_0 + z;
    }

    if (xine_event.type != 0) {
      xine_event.data_length = 0;
      xine_event.data        = NULL;
      xine_event.stream      = gui->stream;
      xitk_gettime_tv (&xine_event.tv);

      xitk_lock (gui->xitk, 1);
      gui->nongui_error_msg = network_err_msg;
      xine_event_send (gui->stream, &xine_event);
      gui->nongui_error_msg = NULL;
      xitk_lock (gui->xitk, 0);
      return 1;
    }
  }
  return 0;
}

static int do_seek (client_info_t *info) {
  gGui_t *gui = info->gui;
  int nargs;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  if ((xine_get_stream_info (gui->stream, XINE_STREAM_INFO_SEEKABLE)) == 0) {
    gui->nongui_error_msg = NULL;
    xitk_lock (gui->xitk, 0);
    return 1;
  }

  nargs = info->command.num_args - 1;
  if (nargs > 0) {
    if (nargs == 1) {
      const char *arg = info->command.buf + info->command.args[1].start;
      int pos;

      if (sscanf (arg, "%%%d", &pos) == 1) {
        if (pos > 100)
          pos = 100;
        else if (pos < 0)
          pos = 0;
        gui_set_current_position (gui, (65535 * pos + 50) / 100);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        return 1;
      } else if ((((arg[0] == '+') || (arg[0] == '-')) && (isdigit (arg[1]))) || (isdigit (arg[0]))) {
        pos = atoi (arg);
        gui_seek_relative (gui, pos);
        gui->nongui_error_msg = NULL;
        xitk_lock (gui->xitk, 0);
        return 1;
      }
    }
  }
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 0;
}

static int do_halt (client_info_t *client_info) {
  gGui_t *gui = client_info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_exit (NULL, gui);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

static void network_messenger(void *data, char *message) {
  if (message) {
    char buf[MAX_LINE];
    size_t l = strlen (message);
    int socket = (int)(intptr_t) data;

    if (l > sizeof (buf) - 2)
      l = sizeof (buf) - 2;
    memcpy (buf, message, l);
    memcpy (buf + l, "\n", 2);
    _sock_write (socket, buf, l + 1);
  }
}

static int do_snap (client_info_t *client_info) {
  gGui_t *gui = client_info->gui;

  xitk_lock (gui->xitk, 1);
  gui->nongui_error_msg = network_err_msg;
  gui_playlist_lock (gui);
  create_snapshot (gui, gui->mmk.mrl,
		  network_messenger, network_messenger, (void *)(intptr_t)client_info->socket);
  gui_playlist_unlock (gui);
  gui->nongui_error_msg = NULL;
  xitk_lock (gui->xitk, 0);
  return 1;
}

static int do_exit (client_info_t *info) {
  if (info) {
    xui_network_remote_t *nr = info->nr;
    if (info->gui->verbosity >= 2)
      printf ("network.remote.server.client.delete (%d).\n", info->index);
    _sock_write (info->socket, "bye!\n", 5);
    nr->fds[info->index] = -1;
    nr->info[info->index] = NULL;
    close (info->socket);
    free (info);
    pthread_mutex_lock (&nr->mutex);
    nr->refs--;
    pthread_mutex_unlock (&nr->mutex);
  }
  return 1;
}

static const commands_t commands[_I_last] = {
  [_I_commands] = { CMD_ARGS_NONE, do_commands,
    "Display all available commands, tab separated, dot terminated.",
    "  commands <no arguments>\n" },
  [_I_help] = { CMD_ARGS_OPTIONAL | CMD_PUBLIC, do_help,
    "Display the help text if a command name is specified, otherwise display "
    "all available commands.",
    "  help <command>\n" },
  [_I_syntax] = { CMD_ARGS_NEED | CMD_PUBLIC, do_syntax,
    "Display a command syntax.",
    "  syntax <command>\n" },
  [_I_identify] = { CMD_ARGS_NEED | CMD_PUBLIC, do_auth,
    "Identify client.",
    "  identify <ident>:<password>\n" },
  [_I_mrl] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_mrl,
    "manage mrls",
    "  mrl add <mrl> <mrl> ...\n"
    "  mrl play <mrl>\n"
    "  mrl next|prev\n" },
  [_I_play] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_play,
    "start playback",
    "  play <no arguments>\n" },
  [_I_playlist] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_playlist,
    "manage playlist",
    "  playlist show|first|prev|next|last|stop|continue\n"
    "  playlist select <num>\n"
    "  playlist delete all|*|<num>\n" },
  [_I_stop] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_stop,
    "stop playback",
    "  stop <no arguments>\n" },
  [_I_pause] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_pause,
    "pause/resume playback",
    "  pause <no arguments>\n" },
  [_I_exit] = { CMD_ARGS_NONE | CMD_PUBLIC, do_exit,
    "close connection",
    "  exit <no arguments>\n" },
  [_I_fullscreen] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_fullscreen,
    "fullscreen toggle",
    "  fullscreen <no arguments>\n" },
#ifdef HAVE_XINERAMA
  [_I_xineramafull] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_xinerama_fullscreen,
    "xinerama fullscreen toggle",
    "  xineramafull <no arguments>\n" },
#endif
  [_I_get] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_get,
    "get values",
    "  get status|speed|position|length|loop\n"
    "  get audio channel|lang|volume|mute\n"
    "  get spu channel|lang|offset\n"
    "  get av offset\n" },
  [_I_set] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_set,
    "set values",
    "  set audio channel <num>\n"
    "  set audio volume <percent>\n"
    "  set audio mute <state>\n"
    "  set spu channel <num>\n"
    "  set spu offset <offset>\n"
    "  set av offset <offset>\n"
    "  set speed <XINE_SPEED_PAUSE|XINE_SPEED_SLOW_4|XINE_SPEED_SLOW_2|XINE_SPEED_NORMAL|XINE_SPEED_FAST_2|XINE_SPEED_FAST_4>\n"
    "            <        |       |        /4       |        /2       |        =        |        *2       |        *4       >\n"
    "  set loop no|loop|repeat|shuffle|shuffle+\n" },
  [_I_gui] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_gui,
    "manage gui windows",
    "  gui hide|output|panel|playlist|control|acontrol|mrl|setup|apost|vpost|help|log\n" },
  [_I_event] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_event,
    "Send an event to xine",
    "  event menu1 .. menu7\n"
    "  event 0 .. 9\n"
    "  event +10\n"
    "  event up|down|left|right|next|previous|select\n"
    "  event angle next|previous\n" },
  [_I_seek] = { CMD_ARGS_NEED | CMD_PUBLIC | CMD_NEED_AUTH, do_seek,
    "Seek in current stream",
    "  seek %0 .. %100\n"
    "  seek +|-<secs>\n" },
  [_I_halt] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_halt,
    "Stop xine program",
    "  halt <no arguments>\n" },
  [_I_snapshot] = { CMD_ARGS_NONE | CMD_PUBLIC | CMD_NEED_AUTH, do_snap,
    "Take a snapshot",
    "  snapshot <no arguments>\n" },
};

/*
 * *********** COMMANDS ENDS **********
 */

/*
 * Handle user entered commands.
 */
static void handle_client_command (client_info_t *info, const char *line) {
  const char *next = line;

  if (info->gui->verbosity >= 2)
    printf ("network.remote.server.client.line (\"%s\").\n", line);

  /* pass command line to the chainsaw */
  while (next[0]) {
    char buf[28 + MAX_LINE + MAX_ARGS * 3 + 1], *e = buf + sizeof (buf) - 8;
    int len = _network_get_line (info, next), i;
    const commands_t *cmd;

    if (!len)
      break;

    if (info->gui->verbosity >= 2) {
      uint32_t n;
      char *p = buf;
      memcpy (p, "network.remote.server.client.command (", 38); p += 38;
      for (n = 0; n < info->command.num_args; n++) {
        *p++ = '"';
        memcpy (p, info->command.buf + info->command.args[n].start, info->command.args[n].len);
        p += info->command.args[n].len;
        memcpy (p, "\" ", 2); p += 2;
      }
      memcpy (p, ")\n", 3);
      printf ("%s", buf);
    }

    i = info->name_list_find (cmd_names, sizeof (cmd_names) / sizeof (cmd_names[0]),
      info->command.buf + info->command.args[0].start, info->command.args[0].len);
    if (i >= 0) {
      cmd = commands + cmd_names[i].index;
      /* flags that _dont_ match the count of user args. */
      uint32_t flags1 = (info->command.num_args > 1) ? CMD_ARGS_NONE : CMD_ARGS_NEED;
      uint32_t flags2 = info->authentified ? 0 : CMD_NEED_AUTH;
      if (cmd->flags & flags1) {
        /* give automatic syntax help when args are missing or ignored, ... */
        _sock_write (info->socket, cmd->syntax, strlen (cmd->syntax));
        cmd = NULL;
      } else if (cmd->flags & flags2) {
        int l = snprintf (buf, e - buf, "You need to be authentified to use '%s' command, use 'identify'.\n\n", cmd_names[i].name);
        _sock_write (info->socket, buf, l);
        cmd = NULL;
      }
    } else {
      /* help */
      info->command.num_args = 1;
      cmd = commands + _I_help;
    }
    if (cmd && !cmd->function (info))
      /* ... or when args were not understood :-) */
      _sock_write (info->socket, cmd->syntax, strlen (cmd->syntax));

    next += len;
  }
}

/*
 *
 */
static void *server_thread (void *data) {
  xui_network_remote_t *nr = data;
  gGui_t *gui = nr->gui;
  char             service[32] = DEFAULT_XINECTL_PORT;
  int              i;
  sigset_t         sigpipe_mask;

  sigemptyset (&sigpipe_mask);
  sigaddset (&sigpipe_mask, SIGPIPE);
  if (pthread_sigmask (SIG_BLOCK, &sigpipe_mask, NULL) == -1)
    perror ("pthread_sigmask");

  do {
    unsigned port = gui->network_port;
    if (!port) {
      /*  Search in /etc/services if a xinectl entry exist */
      struct servent *serv_ent = getservbyname ("xinectl", "tcp");
      if (!serv_ent)
        break;
      port = ntohs (serv_ent->s_port);
    }
    sprintf (service, "%u", port);
  } while (0);

  /* Load passwd file */
  /* password file syntax is:
   *  - one entry per line
   *  - syntax: 'identifier:password' length limit is 16 chars each
   *  - a password beginning by an asterisk '*' lock the user.
   *  - if a line contain 'ALL:ALLOW' (case sensitive),
   *    all client are allowed to execute all commands.
   *  - if a line contain 'ALL:DENY' (case sensitive),
   *    all client aren't allowed to execute a command, except those
   *    who are specified. ex:
   *  - rule 'ALL' is optional.
   *
   *    ALL:DENY
   *    daniel:f1rmb
   *    .......
   *    All users are denied, except 'daniel' (if he give good passwd)
   *
   *
   *    daniel:f1rmb
   *    foo:*mypasswd
   *    .......
   *    All users are denied, 'daniel' is authorized (with passwd), 'foo'
   *    is locked.
   *
   */
  {
    const char  *passwdfile = ".xine/passwd", *home = xine_get_homedir ();
    char         passwdfilename[2048];
    struct stat  st;

    snprintf (passwdfilename, sizeof (passwdfilename), "%s/%s", home, passwdfile);
    if (stat (passwdfilename, &st) < 0) {
      printf ("network.remote.server: ERROR: there is no password file for network access.!\n");
      goto __failed;
    }
    if (!server_load_passwd (nr, passwdfilename)) {
      printf ("network.remote.server: ERROR: unable to load network server password file.!\n");
      goto __failed;
    }
  }

  while (gui->running) {
    client_info_t *info;
    socklen_t      lsin;
    fd_set         rset, eset;
    int            res;

    if (nr->fds[MAX_CLIENTS] < 0) {
      nr->fds[MAX_CLIENTS] = sock_serv (service, "tcp", 5);
      if (nr->fds[MAX_CLIENTS] < 0)
        break;
    }

    do {
      int nfds = nr->break_pipe[0];

      FD_ZERO (&rset);
      FD_ZERO (&eset);
      FD_SET (nr->break_pipe[0], &rset);
      FD_SET (nr->break_pipe[0], &eset);
      for (i = 0; i < MAX_CLIENTS + 1; i++) {
        int fd = nr->fds[i];
        if (fd < 0)
          continue;
        FD_SET (fd, &rset);
        FD_SET (fd, &eset);
        if (fd > nfds)
          nfds = fd;
      }
      res = select (nfds + 1, &rset, NULL, &eset, NULL);
      if (res < 0)
        break;
      if (FD_ISSET (nr->break_pipe[0], &rset) || FD_ISSET (nr->break_pipe[0], &eset)) {
        res = -2;
        break;
      }
    } while (res == 0);
    if (res < 0)
      break;

    /* new client? */
    if (FD_ISSET (nr->fds[MAX_CLIENTS], &rset)) {
      for (i = 0; (i < MAX_CLIENTS) && (nr->fds[i] >= 0); i++) ;
      if (i >= MAX_CLIENTS) {
        /* too many clients */
        close (nr->fds[MAX_CLIENTS]);
        nr->fds[MAX_CLIENTS] = -1;
      } else {
        nr->info[i] = info = (client_info_t *)calloc (1, sizeof (*info));
        if (!info) {
          close (nr->fds[MAX_CLIENTS]);
          nr->fds[MAX_CLIENTS] = -1;
        } else {
          info->gui = gui;
          info->nr = nr;
          info->name_list_find = _name_list_find;
          info->index = i;
          memset (&info->name, 0, sizeof (info->name));
          memset (&info->passwd, 0, sizeof (info->passwd));
          lsin = sizeof (info->fsin.in);
          nr->fds[i] = info->socket = accept (nr->fds[MAX_CLIENTS], &(info->fsin.sa), &lsin);
          if (info->socket < 0) {
            free (info);
            nr->info[i] = NULL;
          } else {
            info->authentified = is_client_authorized (info);
            pthread_mutex_lock (&nr->mutex);
            nr->refs++;
            pthread_mutex_unlock (&nr->mutex);
            if (gui->verbosity >= 2)
              printf ("network.remote.server.client.new (%d).\n", i);
            {
              char buf[320], myfqdn[256];
              size_t l;
              struct hostent *hp = NULL;
              buf[0] = 0;
              myfqdn[0] = 0;
              if (!gethostname (myfqdn, sizeof (myfqdn) - 1))
                hp = gethostbyname (myfqdn);
              l = snprintf (buf, sizeof (buf),
                "%s " PACKAGE " " VERSION " remote server. Nice to meet you.\n", hp ? hp->h_name : "");
              _sock_write (info->socket, buf, l);
            }
          }
          close (nr->fds[MAX_CLIENTS]);
          nr->fds[MAX_CLIENTS] = -1;
        }
      }
    }

    /* client commands? */
    for (i = 0; i < MAX_CLIENTS; i++) {
      char buf[MAX_LINE + 8];
      int len;
      if (nr->fds[i] < 0)
        continue;
      if (!FD_ISSET (nr->fds[i], &rset))
        continue;
      info = nr->info[i];
      len = sock_read (info->socket, buf, sizeof (buf) - 8);
      if (len <= 0) {
        close (info->socket);
        nr->fds[i] = -1;
        nr->info[i] = NULL;
        free (info);
        pthread_mutex_lock (&nr->mutex);
        nr->refs--;
        pthread_mutex_unlock (&nr->mutex);
        if (gui->verbosity >= 2)
          printf ("network.remote.server.client.delete (%d).\n", i);
      } else {
        nr->current = i;
        buf[len] = 0;
        handle_client_command (info, buf);
      }
    }
  }

  for (i = 0; i < MAX_CLIENTS + 1; i++) {
    if (nr->fds[i] >= 0) {
      _sock_write (nr->fds[i], "server has shut down. bye!\n", 27);
      close (nr->fds[i]);
      nr->fds[i] = -1;
    }
  }
  for (i = 0; i < MAX_CLIENTS; i++) {
    free (nr->info[i]);
    nr->info[i] = NULL;
  }

  { /* Freeing passwords */
    int i;

    for (i = 0; nr->passwds[i]; i++)
      free (nr->passwds[i]);
    free (nr->passwds);
  }

 __failed:

  pthread_mutex_lock (&nr->mutex);
  nr->refs--;
  pthread_mutex_unlock (&nr->mutex);

  return NULL;
}

/*
 *
 */
int start_remote_server (gGui_t *gui) {
  xui_network_remote_t *nr;
  int err = 0, i;

  if (!gui)
    return EINVAL;
  if (!gui->network)
    return 0;

  if (gui->verbosity >= 2)
    printf ("network.remote.server.start ().\n");
  if (gui->network_remote)
    return EEXIST;

  gui->network_remote = nr = malloc (sizeof (*gui->network_remote));
  if (!nr)
    return ENOMEM;

  nr->gui = gui;
  nr->break_pipe[0] = nr->break_pipe[1] = -1;
  for (i = 0; i < MAX_CLIENTS + 1; i++)
    nr->fds[i] = -1;
  for (i = 0; i < MAX_CLIENTS; i++)
    nr->info[i] = NULL;
  nr->passwds = NULL;

  nr->refs = 0;

  if (pipe (nr->break_pipe)) {
    err = errno;
    gui->network_remote = NULL;
    free (nr);
    return err;
  }
  fcntl (nr->break_pipe[0], F_SETFL, fcntl (nr->break_pipe[0], F_GETFL));
  fcntl (nr->break_pipe[1], F_SETFL, fcntl (nr->break_pipe[1], F_GETFL));

  nr->refs = 1;
  pthread_mutex_init (&nr->mutex, NULL);
  pthread_mutex_lock (&nr->mutex);
  if (pthread_create (&gui->network_remote->server, NULL, server_thread, nr)) {
    err = errno;
    pthread_mutex_unlock (&nr->mutex);
    pthread_mutex_destroy (&nr->mutex);
    gui->network_remote = NULL;
    free (nr);
    return err;
  }
  pthread_mutex_unlock (&nr->mutex);
  if (gui->verbosity >= 2)
    printf ("network.remote.server.start (OK).\n");
  return 0;
}

int stop_remote_server (gGui_t *gui) {
  xui_network_remote_t *nr;
  char buf[8] = { [0] = 0 };
  int refs;

  if (!gui)
    return 0;

  if (gui->verbosity >= 2)
    printf ("network.remote.server.stop ().\n");

  nr = gui->network_remote;
  gui->network_remote = NULL;
  if (!nr)
    return 0;

  pthread_mutex_lock (&nr->mutex);
  refs = nr->refs;
  pthread_mutex_unlock (&nr->mutex);

  write (nr->break_pipe[1], buf, 1);
  {
    void *dummy;
    pthread_join (nr->server, &dummy);
  }
  read (nr->break_pipe[0], buf, 8);

  pthread_mutex_destroy (&nr->mutex);
  close (nr->break_pipe[1]);
  close (nr->break_pipe[0]);
  free (nr);
  if (gui->verbosity >= 2)
    printf ("network.remote.server.stop (%d).\n", refs);
  return 0;
}

/** /server only ********************************************************/
#endif

#endif  /* HAVE_READLINE */
