/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * Unique public header file for xitk µTK.
 *
 */

#ifndef _XITK_H_
#define _XITK_H_

#ifndef PACKAGE_NAME
#error config.h not included
#endif

#include <stddef.h>
#include <stdlib.h>
#include <limits.h>
#include <stdint.h>

#define XITK_MAJOR_VERSION          (0)
#define XITK_MINOR_VERSION          (11)
#define XITK_SUB_VERSION            (0)
#define XITK_VERSION                "0.11.0"

#define XITK_CHECK_VERSION(major, minor, sub)                          \
                                    (XITK_MAJOR_VERSION > (major) ||   \
                                     (XITK_MAJOR_VERSION == (major) && \
                                      XITK_MINOR_VERSION > (minor)) || \
                                     (XITK_MAJOR_VERSION == (major) && \
                                      XITK_MINOR_VERSION == (minor) && \
                                      XITK_SUB_VERSION >= (sub)))

#define XITK_WIDGET_MAGIC           0x7869746b

#ifndef NAME_MAX
#define XITK_NAME_MAX               256
#else
#define XITK_NAME_MAX               NAME_MAX
#endif

#ifndef PATH_MAX
#define XITK_PATH_MAX               768
#else
#define XITK_PATH_MAX               PATH_MAX
#endif

/* optimize away .foo = NULL; assignments after calloc (). */
#if defined(__GNUC__) || defined(__clang__)
#  define xitk_init_NULL() ({ \
  const union { char b[sizeof (void *)]; void *p; } t = { .b = { [0] = 0 }}; \
  t.p != NULL; \
})
#else
static inline int xitk_init_NULL (void) {
  const union { char b[sizeof (void *)]; void *p; } t = { .b = { [0] = 0 }};
  return t.p != NULL;
}
#endif

/* paranoia #1: error if *cont_ptr does not have an element elem_name.
 * paranoia #2: warn if it has wrong type. */
#define xitk_container(cont_ptr,elem_ptr,elem_name) do { \
  const typeof (elem_ptr) xc_dummy = &((const typeof (cont_ptr))0)->elem_name; \
  cont_ptr = (elem_ptr) ? (typeof (cont_ptr))(void *)((uint8_t *)(elem_ptr) - (uintptr_t)xc_dummy) : NULL; \
} while (0)

/** a branchless version of ((_val & _from) ? _to : 0). */
#define xitk_bitmove(_val,_from,_to) \
  ((_from > _to) ? ((uint32_t)(_val) & (_from)) / ((_from) / (_to)) : ((uint32_t)(_val) & (_from)) * ((_to) / (_from)))

size_t xitk_lower_strlcpy (char *dest, const char *src, size_t dlen);

typedef union { uint8_t b[4]; uint32_t v; } xitk_find_byte_mask_t;
extern const xitk_find_byte_mask_t xitk_find_byte_mask[4];
extern const uint8_t xitk_find_byte_rest[32];

struct timeval;
struct timespec;
int xitk_gettime_tv (struct timeval *tv);
int xitk_gettime_ts (struct timespec *ts);

static inline uint32_t xitk_find_byte (const char *s, uint32_t byte) {
  const uint32_t eor = ~((byte << 24) | (byte << 16) | (byte << 8) | byte);
  const uint32_t left = (uintptr_t)s & 3;
  const uint32_t *p = (const uint32_t *)(s - left);
  const union {
    uint32_t v;
    uint8_t b[4];
  } endian = {16};
  uint32_t w = (*p++ ^ eor) & xitk_find_byte_mask[left].v;
  while (1) {
    w = w & 0x80808080 & ((w & 0x7f7f7f7f) + 0x01010101);
    if (w)
      break;
    w = *p++ ^ eor;
  }
  /* bits 31, 23, 15, 7 -> 3, 2, 1, 0 */
  w = (w * 0x00204081) & 0xffffffff;
  w >>= 28;
  return ((const char *)p - s) - xitk_find_byte_rest[endian.b[0] + w];
}

static inline uint32_t xitk_find_0_or_byte (const char *s, uint32_t byte) {
  const uint32_t eor0 = ~0u;
  const uint32_t eor1 = ~((byte << 24) | (byte << 16) | (byte << 8) | byte);
  const uint32_t left = (uintptr_t)s & 3;
  const uint32_t *p = (const uint32_t *)(s - left);
  const union {
    uint32_t v;
    uint8_t b[4];
  } endian = {16};
  uint32_t w0 = (*p ^ eor0) & xitk_find_byte_mask[left].v;
  uint32_t w1 = (*p++ ^ eor1) & xitk_find_byte_mask[left].v;
  while (1) {
    w0 = (w0 & 0x80808080 & ((w0 & 0x7f7f7f7f) + 0x01010101))
       | (w1 & 0x80808080 & ((w1 & 0x7f7f7f7f) + 0x01010101));
    if (w0)
      break;
    w0 = *p ^ eor0;
    w1 = *p++ ^ eor1;
  }
  /* bits 31, 23, 15, 7 -> 3, 2, 1, 0 */
  w0 = (w0 * 0x00204081) & 0xffffffff;
  w0 >>= 28;
  return ((const char *)p - s) - xitk_find_byte_rest[endian.b[0] + w0];
}

typedef struct {
  char *s, buf[64];
  size_t len;
} xitk_short_string_t;
void xitk_short_string_init (xitk_short_string_t *s);
#define xitk_short_string_init(_s) (_s)->s = (_s)->buf, (_s)->buf[0] = 0, (_s)->len = 0
/* return strlen (v), or (size_t)-1 if string is unchanged. */
size_t xitk_short_string_set (xitk_short_string_t *s, const char *v);
void xitk_short_string_deinit (xitk_short_string_t *s);
#define xitk_short_string_deinit(_s) if ((_s)->s != (_s)->buf) { free ((_s)->s); (_s)->s = NULL; (_s)->len = 0; }

/* refcounted shared strings. */
const char *xitk_ref_string_ref (const char *str);
int xitk_ref_string_unref (const char **str);

/* the versioned string feature. */
typedef struct {
  char *s;
  unsigned int bsize;   /** << if not 0, s points to a user allocated buf of this size. */
  int          version; /** << how many changes since init. */
} xitk_vers_string_t;
void xitk_vers_string_init (xitk_vers_string_t *vs, char *user_buf, size_t user_bsize);
/* return whether or not there was a change. */
int xitk_vers_string_set (xitk_vers_string_t *vs, const char *s);
int xitk_vers_string_get (xitk_vers_string_t *to, const xitk_vers_string_t *from);
void xitk_vers_string_deinit (xitk_vers_string_t *vs);

void xitk_usec_sleep (unsigned long);

/* this will pad the return with 8 zero bytes both before and after.
 * filesize, if set, is the size limit before and the actual size after the call. */
char *xitk_cfg_load (const char *filename, size_t *filesize);
void xitk_cfg_unload (char *buf);

typedef struct {
  /* 0 for root node tree[0] */
  short int level;
  /* byte length of key */
  unsigned short int klen;
  /* array index. */
  int next, prev, parent, first_child, last_child;
  /* root node: total count of nodes.
   * otherwise: offset into contents, converted to lowercase, or -1. */
  int key;
  /* offset into contents, or -1. */
  int value;
} xitk_cfg_parse_t;

#define XITK_CFG_PARSE_DEBUG 1 /* show extra info */
#define XITK_CFG_PARSE_CASE 2 /* dont lowercase keys */
xitk_cfg_parse_t *xitk_cfg_parse (char *contents, int flags);
void xitk_cfg_unparse (xitk_cfg_parse_t *tree);

/* supports "#<hex>", "0x<hex>". "0<octal", "<decimal>". */
int32_t xitk_str2int32 (const char **str);

/* supports some real names, "#rrggbb", "0xrrggbb".
 * returns (red << 16) | (green << 8) | blue, or ~0u. */
uint32_t xitk_get_color_name (const char *color);
/* special color values for xitk_*_noskin_create () */
#define XITK_NOSKIN_DEFAULT   0x80000000
#define XITK_NOSKIN_TEXT_NORM 0x80000001
#define XITK_NOSKIN_TEXT_INV  0x80000002

typedef struct xitk_s xitk_t;
typedef struct xitk_backend_s xitk_backend_t;
typedef struct xitk_be_display_s xitk_be_display_t;
typedef struct xitk_be_image_s xitk_be_image_t;
typedef struct xitk_be_font_s xitk_be_font_t;
typedef struct xitk_be_window_s xitk_be_window_t;

/** use this before messing with widgets directly from a thread outside xitk_run ():
 *  mode = 0 (unlock), 1 (lock), 2 (try immediate lock), > 2 (wait at most (mode - 2) milliseconds for lock).
 *  return 0 (ok) or Exxx. */
int xitk_lock (xitk_t *_xitk, int mode);

typedef struct xitk_widget_s xitk_widget_t;
typedef struct xitk_widget_list_s xitk_widget_list_t;
typedef struct xitk_skin_config_s xitk_skin_config_t;
typedef struct xitk_skin_element_info_s xitk_skin_element_info_t;
typedef struct xitk_be_font_s xitk_font_t;
typedef struct xitk_image_s xitk_image_t;
typedef struct xitk_window_s xitk_window_t;

typedef void (*xitk_startup_callback_t)(void *);
typedef void (*xitk_simple_callback_t)(xitk_widget_t *, void *);
typedef void (*xitk_state_callback_t)(xitk_widget_t *, void *, int);
typedef void (*xitk_ext_state_callback_t)(xitk_widget_t *, void *, int, int modifiers);
typedef void (*xitk_state_double_callback_t)(xitk_widget_t *, void *, double);
typedef void (*xitk_string_callback_t)(xitk_widget_t *, void *, const char *);
typedef void (*xitk_dnd_callback_t) (void *data, const char *filename);

typedef struct {
  int x, y, width, height;
} xitk_rect_t;

/*
 * Signal handler callback function.
 * Xitk will call this function when a signal happens.
 * return 1 (handled), 0 (not handled).
 */
typedef int (*xitk_signal_callback_t) (int signr, void *userdata);

/*
 * A unique key returned by register function.
 * It is necessary to store it at program side,
 * because it will be necessary to pass it for
 * identification of caller.
 */
typedef int xitk_register_key_t;

int xitk_widget_key_event (xitk_widget_t *w, const char *string, int modifier, int key_up);

/*
 *
 */

typedef struct {
  int x, y;
} xitk_point_t;

typedef struct {
  int first, last;
} xitk_range_t;

typedef struct {
  xitk_window_t                    *xwin;
  int                               x;
  int                               y;
  int                               height;
  int                               width;
  char                              name[64];
} window_info_t;

typedef struct {
  xitk_image_t *image;
  int x, y, width, height;
  /* 0 (default), 1..6 (normal, focus, selected, focus+selected, disabled, disabled+selected) */
  int num_states;
} xitk_part_image_t;

#define WIDGET_GROUP                0x00000800 /** << Group of widgets widget */
#define WIDGET_GROUP_MEMBER         0x80000000 /** << Grouped widget, itself */
#define WIDGET_TABABLE              0x40000000 /** << Does widget take part in tab cycle */
#define WIDGET_FOCUSABLE            0x20000000 /** << Is widget focusable */
#define WIDGET_CLICKABLE            0x10000000 /** << Is widget clickable */
#define WIDGET_KEEP_FOCUS           0x08000000 /** << Widget keeps focus after tab/click */
#define WIDGET_KEYABLE              0x04000000 /** << Widget support key events */
#define WIDGET_PARTIAL_PAINTABLE    0x02000000 /** << Widget support partial repaint */
#define WIDGET_WINDOW_FOCUSABLE     0x01000000 /** << Windget followes window focus */

/* Grouped widgets */
#define WIDGET_GROUP_MASK           0x00fff000
#define WIDGET_GROUP_BROWSER        0x00001000
#define WIDGET_GROUP_MRLBROWSER     0x00002000
#define WIDGET_GROUP_COMBO          0x00004000
#define WIDGET_GROUP_TABS           0x00008000
#define WIDGET_GROUP_INTBOX         0x00010000
#define WIDGET_GROUP_DOUBLEBOX      0x00020000
#define WIDGET_GROUP_MENU           0x00040000
#define WIDGET_GROUP_BUTTON_LIST    0x00080000

#define WIDGET_TYPE_MASK            0x00000fff
/* Group leaders.. */
#define WIDGET_TYPE_COMBO           0x00000801
#define WIDGET_TYPE_DOUBLEBOX       0x00000802
#define WIDGET_TYPE_INTBOX          0x00000803
#define WIDGET_TYPE_BROWSER         0x00000804
#define WIDGET_TYPE_MRLBROWSER      0x00000805
#define WIDGET_TYPE_TABS            0x00000806
#define WIDGET_TYPE_MENU            0x00000807
#define WIDGET_TYPE_BUTTON_LIST     0x00000808
/* Real widgets. */
#define WIDGET_TYPE_BUTTON          0x00000001
#define WIDGET_TYPE_LABELBUTTON     0x00000002
#define WIDGET_TYPE_LABEL           0x00000003
#define WIDGET_TYPE_SLIDER          0x00000004
#define WIDGET_TYPE_CHECKBOX        0x00000005
#define WIDGET_TYPE_IMAGE           0x00000006
#define WIDGET_TYPE_INPUTTEXT       0x00000007

/* See */
#define ALIGN_UNSET                 0
#define ALIGN_LEFT                  1
#define ALIGN_CENTER                2
#define ALIGN_RIGHT                 3
#define ALIGN_DEFAULT               (ALIGN_LEFT)

/*
 * See xitk_get_modifier_key()
 */
#define MODIFIER_NOMOD              0x00000000
#define MODIFIER_SHIFT              0x00000001 /** << "shift" key, left or right. */
#define MODIFIER_LOCK               0x00000002 /** << should be "caps lock on", but seems to be always 0. */
#define MODIFIER_CTRL               0x00000004 /** << "ctrl" key, left or right. */
#define MODIFIER_META               0x00000008 /** << "alt" key, left or right. */
#define MODIFIER_NUML               0x00000010 /** << should be "num lock on", but seems to be always 0. */
#define MODIFIER_MOD3               0x00000020 /** << any of the first 3 mouse buttons held (why?) */
#define MODIFIER_MOD4               0x00000040 /** << "w******s" key, left or right. */
#define MODIFIER_MOD5               0x00000080 /** << "alt-gr" key. */
#define MODIFIER_BUTTON1            0x00000100 /** << left mouse button held. */
#define MODIFIER_BUTTON2            0x00000200 /** << right mouse button held. */
#define MODIFIER_BUTTON3            0x00000400 /** << middle mouse button held. */
#define MODIFIER_BUTTON4            0x00000800
#define MODIFIER_BUTTON5            0x00001000
#define MODIFIER_MOUSE_MOVE         0x00100000 /** << since last enter/move/click/leave event. */
#define MODIFIER_MOUSE_OUTSIDE      0x00200000 /** << really outside window, or over a transparent or covered part. */

/*
 * Result of dialog window
 */
#define XITK_WINDOW_ANSWER_UNKNOWN  0
#define XITK_WINDOW_ANSWER_OK       1
#define XITK_WINDOW_ANSWER_YES      2
#define XITK_WINDOW_ANSWER_NO       3
#define XITK_WINDOW_ANSWER_CANCEL   4

/*
 * See xitk_get_wm_type()
 */
#define WM_TYPE_COMP_MASK           0x00007FFF
#define WM_TYPE_UNKNOWN             0x00000000
#define WM_TYPE_GNOME_COMP          0x80000000
#define WM_TYPE_EWMH_COMP           0x40000000
#define WM_TYPE_KWIN                0x00000001
#define WM_TYPE_E                   0x00000002
#define WM_TYPE_ICE                 0x00000003
#define WM_TYPE_WINDOWMAKER         0x00000004
#define WM_TYPE_MOTIF               0x00000005
#define WM_TYPE_XFCE                0x00000006
#define WM_TYPE_SAWFISH             0x00000007
#define WM_TYPE_METACITY            0x00000008
#define WM_TYPE_AFTERSTEP           0x00000009
#define WM_TYPE_BLACKBOX            0x0000000A
#define WM_TYPE_LARSWM              0x0000000B
#define WM_TYPE_DTWM                0x0000000C

typedef enum {
  WINDOW_TYPE_NONE = 0,
  WINDOW_TYPE_DESKTOP,
  WINDOW_TYPE_DOCK,
  WINDOW_TYPE_TOOLBAR,
  WINDOW_TYPE_MENU,
  WINDOW_TYPE_UTILITY,
  WINDOW_TYPE_SPLASH,
  WINDOW_TYPE_DIALOG,
  //WINDOW_TYPE_DROPDOWN_MENU,
  //WINDOW_TYPE_POPUP_MENU,
  //WINDOW_TYPE_TOOLTIP,
  //WINDOW_TYPE_NOTIFICATION,
  //WINDOW_TYPE_COMBO,
  WINDOW_TYPE_DND,
  WINDOW_TYPE_NORMAL,
  WINDOW_TYPE_END
} xitk_wm_window_type_t;

void xitk_window_set_wm_window_type (xitk_window_t *w, xitk_wm_window_type_t type);

typedef struct {
  xitk_widget_list_t  *wl;
  xitk_widget_t       *group;               /** << an optional widget that completely covers this one
                                             *     (eg an actual group widget, or a background image). */
  void                *userdata;            /** << will be passed to callbacks. */
  const char          *skin_element_name;   /** << XITK_NOSKIN_* or NULL for not skinned widgets. */
  const char          *tips;                /** << NULL if no tips yet. */
  unsigned int         add_state;           /** << if XITK_WIDGET_STATE_*, the widget will be added to list. */
  unsigned int         mode_mask;
  unsigned int         mode_value;          /** << see xitk_widget_mode (). */
} xitk_new_widget_t;

/* *******
 * INIT: widget lib initialization and friends
 */

/** Register a callback function called when a signal heppens. */
void xitk_register_signal_handler (xitk_t *xitk, xitk_signal_callback_t sigcb, void *user_data);

/** Remove widgetkey_t entry in internal table. */
void xitk_unregister_event_handler (xitk_t *xitk, xitk_register_key_t *key);

/** Copy window information matching with key in passed window_info_t struct. */
int xitk_get_window_info (xitk_t *xitk, xitk_register_key_t key, window_info_t *winf);

/** Find a window by safe key. */
xitk_window_t *xitk_get_window (xitk_t *xitk, xitk_register_key_t key);

int       xitk_window_get_backend_type(xitk_window_t *xwin);
uintptr_t xitk_window_get_native_id(xitk_window_t *xwin);
uintptr_t xitk_window_get_native_display_id(xitk_window_t *xwin);

/** Initialization function, should be the first call to widget lib. */
xitk_t *xitk_init (const char *prefered_visual, int install_colormap,
  int use_x_lock_display, int use_synchronized_x, int verbosity);

void xitk_free (xitk_t **);

const char *xitk_set_locale(void);

long int xitk_reset_screen_saver(xitk_t *xitk, long int timeout);
int xitk_change_video_mode(xitk_t *xitk, xitk_window_t *w, int min_width, int min_height);

void xitk_get_display_size (xitk_t *xitk, int *w, int *h);
double xitk_get_display_ratio (xitk_t *xitk);

int xitk_get_layer_level(xitk_t *xitk);

/** Return WM_TYPE_* */
uint32_t xitk_get_wm_type (xitk_t *xitk);

void xitk_window_set_window_layer(xitk_window_t *w, int layer);
void xitk_window_set_layer_above(xitk_window_t *w);

/** Run the xitk engine. Will only return after a widget_stop() call. */
void xitk_run (xitk_t *xitk, void (* start_cb)(void *data), void *start_data,
  void (* stop_cb)(void *data), void *stop_data);

/** Terminate the xitk engine from within a widget callback.
 * Other functions of the lib shouldn't be called after this one.
 */
void xitk_stop (xitk_t *xitk);

/** Some user settings values. */
typedef enum {
  XITK_SYSTEM_FONT = 1,
  XITK_DEFAULT_FONT,
  XITK_XMB_ENABLE,
  XITK_SHM_ENABLE,
  XITK_MENU_SHORTCUTS_ENABLE,
  XITK_BLACK_COLOR,
  XITK_DISABLED_BLACK_COLOR,
  XITK_WHITE_COLOR,
  XITK_DISABLED_WHITE_COLOR,
  XITK_BG_COLOR,
  XITK_BG_GRAY,
  XITK_FOCUS_COLOR,
  XITK_FOCUS_GRAY,
  XITK_SELECT_COLOR,
  XITK_SEL_FOCUS_COLOR,
  XITK_WARN_BG_COLOR,
  XITK_WARN_FG_COLOR,
  XITK_BAR_STYLE,
  XITK_CHECK_STYLE,
  XITK_CURSORS_FEATURE,
  XITK_TIPS_TIMEOUT,
  XITK_TIMER_LABEL_ANIM,
  XITK_DBL_CLICK_TIME,
  XITK_CFG_END
} xitk_cfg_item_t;

const char *xitk_get_cfg_string (xitk_t *xitk, xitk_cfg_item_t item);
int xitk_get_cfg_num (xitk_t *xitk, xitk_cfg_item_t item);

char *xitk_filter_filename(const char *name);
int xitk_is_dbl_click (xitk_t *xitk, const struct timeval *t1, const struct timeval *t2);


/*
 *
 ****** */

int xitk_get_bool_value (const char *val);

/** Allocate an clean memory of "size" bytes. */
void *xitk_xmalloc(size_t);


typedef struct {
  int x1, x2, y1, y2;
} xitk_hull_t;

/** Behold the master of partial arts ;-) hull may be NULL. */
int xitk_partial_paint_widget_list (xitk_widget_list_t *wl, xitk_hull_t *hull);

/** Return the focused widget. */
xitk_widget_t *xitk_get_focused_widget (xitk_widget_list_t *);

/** Target mouse and tab focus to another widget.
 *  Return 0 (OK), 1 (would make an infinite loop), 2 (invalid widget). */
int xitk_widget_set_focus_redirect (xitk_widget_t *w, xitk_widget_t *focus_redirect);

/** Force the focus to given widget. Return 0 (fail), 1 (OK), 2 (no change). */
int xitk_set_focus_to_widget (xitk_widget_t *w);

/** Return width (in pixel) of widget. */
int xitk_get_widget_width(xitk_widget_t *);

/** Return height (in pixel) of widget. */
int xitk_get_widget_height(xitk_widget_t *);

/** Set position of a widget. */
int xitk_set_widget_pos(xitk_widget_t *w, int x, int y);

/** Get position of a widget. */
int xitk_get_widget_pos(xitk_widget_t *w, int *x, int *y);

uint32_t xitk_get_widget_type (xitk_widget_t *w);
/** makes this widget reflect the window focus instead of the tab/mouse one.
  * also, use the (possibly not shown) window title as tips when they were set to "". */
void xitk_widget_set_window_focusable (xitk_widget_t *w);

/** use XITK_INT_KEEP to just query. */
int xitk_widget_select (xitk_widget_t *w, int index);

#ifdef YET_UNUSED
/** Set widgets of widget list visible. */
void xitk_show_widgets (xitk_widget_list_t *wl, int draw);
/** Set widgets of widget list not visible. */
void xitk_hide_widgets (xitk_widget_list_t *wl);
#endif

xitk_image_t *xitk_get_widget_foreground_skin (xitk_widget_t *w);

/** flags for xitk_new_widget_t.add_state... */
#define XITK_WIDGET_STATE_CLEAR 0x80000000 /** << add and reset. */
#define XITK_WIDGET_STATE_KEEP  0x40000000 /** << add and keep already set state bits. */
/** ...and also for xitk_widgets_state (). */
#define XITK_WIDGET_STATE_ENABLE         1 /** << widget responds to user input. */
#define XITK_WIDGET_STATE_VISIBLE        2 /** << widget is shown. */
#define XITK_WIDGET_STATE_CLICK          4 /** << left mouse button held on this. */
#define XITK_WIDGET_STATE_ON             8 /** << widget is "on" (radio button etc.). */
#define XITK_WIDGET_STATE_IMMEDIATE     16 /** << fire on mouse/enter/space press instead of release. */
#define XITK_WIDGET_STATE_TOGGLE        32 /** << widget can be "on". */
/** modify some state flags. returns the new state of last done widget. */
unsigned int xitk_widgets_state (xitk_widget_t * const *w, unsigned int n, unsigned int mask, unsigned int state);
#define xitk_is_widget_enabled(_w) (!!(xitk_widgets_state (&(_w), 1, 0, 0) & XITK_WIDGET_STATE_ENABLE))
/** unlink and discard widgets. */
void xitk_widgets_delete (xitk_widget_t **w, unsigned int n);

#define XITK_TIPS_STRING_KEEP ((const char *)1)
#define XITK_TIPS_TIMEOUT_OFF 0
#define XITK_TIPS_TIMEOUT_AUTO 1
/* other values are milliseconds. */
void xitk_set_tips_timeout (xitk_t *xitk, unsigned int timeout);
void xitk_set_widget_tips_and_timeout (xitk_widget_t *w, const char *str, unsigned int timeout);
#define xitk_set_widget_tips(w,s) xitk_set_widget_tips_and_timeout (w, s, XITK_TIPS_TIMEOUT_AUTO)

/** cycle through a subset of widgets by xitk key code manually. */
int xitk_widgets_cycle (xitk_widget_t * const *list, uint32_t n, uint32_t xitk_key_arrow);
/*
 * *** Image
 */

/** -2: return max; -1: return current; 0..max: return redraw_needed. */
int xitk_image_quality (xitk_t *xitk, int qual);

/**
 * Load image and return a xitk_image_t data type.
 * if (data != NULL):
 * dsize > 0: decode (down)loaded image file contents,
 *            and aspect preserving scale to fit w and/or h if set.
 * dsize == 0: read from named file, then same as above.
 * dsize == -1: use raw data as (w, h). */
xitk_image_t *xitk_image_new (xitk_t *xitk, const char *data, int dsize, int w, int h);
void xitk_image_ref (xitk_image_t *img);
void xitk_image_set_pix_font (xitk_image_t *image, const char *format);
int xitk_image_inside (xitk_image_t *img, int x, int y);
void xitk_image_copy_rect (xitk_image_t *from, xitk_image_t *to, int x1, int y1, int w, int h, int x2, int y2);

int xitk_image_width(xitk_image_t *);
int xitk_image_height(xitk_image_t *);

void xitk_image_copy (xitk_image_t *from, xitk_image_t *to);

/** Font manipulations. */
xitk_font_t *xitk_font_load_font (xitk_t *xitk, const char *font);
void xitk_font_unload_font (xitk_font_t *xtfs);
void xitk_image_draw_string (xitk_image_t *img, xitk_font_t *xtfs, int x, int y, const char *text, size_t nbytes, int color);
int xitk_font_get_text_width (xitk_font_t *xtfs, const char *c, int nbytes);
int xitk_font_get_string_length (xitk_font_t *xtfs, const char *c);
#ifdef YET_UNUSED
int xitk_font_get_char_width(xitk_font_t *xtfs, const char *c, int maxnbytes, int *nbytes);
int xitk_font_get_text_height(xitk_font_t *xtfs, const char *c, int nbytes);
int xitk_font_get_ascent(xitk_font_t *xtfs, const char *c);
int xitk_font_get_descent(xitk_font_t *xtfs, const char *c);
int xitk_font_get_char_height(xitk_font_t *xtfs, const char *c, int maxnbytes, int *nbytes);
#endif
int xitk_font_get_string_height(xitk_font_t *xtfs, const char *c);
void xitk_font_string_extent (xitk_font_t *xtfs, const char *c,
  int *lbearing, int *rbearing, int *width, int *ascent, int *descent);

/** get a display specific color id (eg a palette index) suitable for rendering.
 *  NOTE: this may return the same as rgb on 24bit display, but we still need to
 *  call this or rendering fails silently sometimes. */
uint32_t xitk_color_db_get (xitk_t *_xitk, uint32_t rgb);
void xitk_color_db_flush (xitk_t *_xitk);

/** xitk image */
xitk_image_t *xitk_image_create_image_with_colors_from_string (xitk_t *xitk,
  const char *fontname, int width, int pad_x, int pad_y, int align, const char *str,
  unsigned int foreground, unsigned int background);
xitk_image_t *xitk_image_from_string(xitk_t *xitk,
  const char *fontname, int width, int align, const char *str);
void xitk_image_draw_image (xitk_widget_list_t *wl, xitk_image_t *im,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y, int sync);
void xitk_part_image_draw (xitk_widget_list_t *wl, xitk_part_image_t *origin, xitk_part_image_t *copy,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y);
void xitk_part_image_copy (xitk_widget_list_t *wl, xitk_part_image_t *from, xitk_part_image_t *to,
  int src_x, int src_y, int width, int height, int dst_x, int dst_y);
/** Free an image object, and return the count of remaining refs. */
int xitk_image_free_image (xitk_image_t **src);

void xitk_image_draw_line (xitk_image_t *img, int x0, int y0, int x1, int y1, unsigned color);
void xitk_image_fill_polygon (xitk_image_t *img, const xitk_point_t *points, int npoints, unsigned color);
void xitk_image_draw_rectangle (xitk_image_t *img, int x, int y, int w, int h, unsigned int color);
void xitk_image_fill_rectangle (xitk_image_t *img, int x, int y, int w, int h, unsigned int color);

/** style: see XITK_DRAW_*. */
void xitk_image_draw_bevel_style (xitk_image_t *img, uint32_t style);
void xitk_part_image_draw_bevel_style (xitk_part_image_t *pi, uint32_t style);
void xitk_image_draw_bevel_two_state (xitk_image_t *img);

void xitk_image_draw_tab (xitk_image_t *img);

void xitk_image_draw_paddle_three_state (xitk_image_t *img, int width, int height);
void xitk_image_draw_paddle_three_state_vertical (xitk_image_t *img);
void xitk_image_draw_paddle_three_state_horizontal (xitk_image_t *img);
void xitk_image_draw_paddle_rotate (xitk_image_t *img);

void xitk_image_draw_rotate_button (xitk_image_t *img);

typedef enum {
  XITK_SYMBOL_USER = 0, /** << xitk_noskin_button_create (): non shared, user repainted skin. */
  XITK_SYMBOL_NONE,
  XITK_SYMBOL_UP,
  XITK_SYMBOL_DOWN,
  XITK_SYMBOL_LEFT,
  XITK_SYMBOL_RIGHT,
  XITK_SYMBOL_MINUS,
  XITK_SYMBOL_PLUS,
  XITK_SYMBOL_CHECK,
  XITK_SYMBOL_LAST
} xitk_symbol_t;

/** type == XITK_SYMBOL_{UP...PLUS}. */
void xitk_image_draw_symbol (xitk_image_t *img, xitk_symbol_t type);

/** style: see XITK_DRAW_{INNER|OUTTER|FLATTER}. */
void xitk_image_draw_relief (xitk_image_t *img, int w, int h, uint32_t style);

#define XITK_DRAW_BEVEL     0
#define XITK_DRAW_INNER     1
#define XITK_DRAW_OUTTER    2
#define XITK_DRAW_FLATTER   3
/* with XITK_DRAW_INNER/XITK_DRAW_OUTTER */
#define XITK_DRAW_DOUBLE    8
#define XITK_DRAW_LIGHT    16
/* bg_color modification */
#define XITK_DRAW_B   0x10000
#define XITK_DRAW_G   0x20000
#define XITK_DRAW_R   0x40000
#define XITK_DRAW_SAT(_sat255) ((_sat255) << 24)
void xitk_image_draw_rectangular_box (xitk_image_t *img, int x, int y, int width, int height, uint32_t style);
/** return rgb background color */
uint32_t xitk_image_draw_frame (xitk_image_t *img, const char *title, const char *fontname,
    int x, int y, int w, int h, uint32_t type);

void xitk_image_draw_checkbox_check (xitk_image_t *img);

/** Windows */
#define XITK_WINDOW_BG_SIMPLE NULL
#define XITK_WINDOW_BG_FRAME ((xitk_image_t *)1)
#define XITK_WINDOW_POS_CENTER (-1)
xitk_window_t *xitk_window_create_window_ext (xitk_t *xitk, int x, int y, int width, int height,
    const char *title, const char *res_name, const char *res_class,
    int override_redirect, int layer_above, xitk_image_t *icon, xitk_image_t *bg_image);

/**
 * Get widget list of window.
 *  - list is created during first call.
 *  - list is automatically released when window is destroyed.
 */
xitk_widget_list_t *xitk_window_widget_list (xitk_window_t *);

void xitk_window_destroy_window (xitk_window_t *w);

#define XITK_WINF_VISIBLE    0x0001 /** << contents is shown on screen */
#define XITK_WINF_ICONIFIED  0x0002 /** << window shown as icon or taskbar entry */
#define XITK_WINF_DECORATED  0x0004 /** << window manager frame around */
#define XITK_WINF_TASKBAR    0x0008 /** << do appear in taskbar */
#define XITK_WINF_PAGER      0x0010 /** << do appear in alt-tab */
#define XITK_WINF_MAX_X      0x0020 /** << maximized width */
#define XITK_WINF_MAX_Y      0x0040 /** << maximized height */
#define XITK_WINF_FULLSCREEN 0x0080 /** << full screen size */
#define XITK_WINF_FOCUS      0x0100 /** << input goes here */
#define XITK_WINF_OVERRIDE_REDIRECT 0x0200 /** << window manager please dont interfere */
#define XITK_WINF_FIXED_POS  0x0400 /** << user may _not_ click move */
#define XITK_WINF_FENCED_IN  0x0800 /** << always stay on screen cmpletely */
#define XITK_WINF_DND        0x1000 /** << do receive drag and drop events */
#define XITK_WINF_GRAB_POINTER 0x2000 /** << Grab mouse pointer */
#define XITK_WINF_LOCK_OPACITY 0x4000

/** try to set the flags in mask to value, return new flags. */
uint32_t xitk_window_flags (xitk_window_t *xwin, uint32_t mask, uint32_t value);

typedef enum {
  XITK_WR_HELPER = 0, /** << stay in front of main, show/hide with vice, default. */
  XITK_WR_ROOT,       /** << main using root window */
  XITK_WR_MAIN,       /** << represent the application. */
  XITK_WR_VICE,       /** << stay in front of main, be main while main is invisible. */
  XITK_WR_SUBMENU     /** << stay in front, keep transient_for parent. */
} xitk_window_role_t;
void xitk_window_set_role (xitk_window_t *xwin, xitk_window_role_t role);

/** return 0 (OK), 1 (window error), 2 (skin error). */
int xitk_window_change_skin (xitk_window_t *xwin, xitk_skin_config_t *skin, const char *background_skin_element_name);

#define XITK_INT_KEEP 0x7fffffff
void xitk_window_move_resize (xitk_window_t *xwin, const xitk_rect_t *r);

void xitk_window_set_input_focus (xitk_window_t *w);

/** Return current background image */
xitk_image_t *xitk_window_get_background_image (xitk_window_t *w);

void xitk_window_apply_background (xitk_window_t *w);

/** Change background. */
int xitk_window_set_background_image (xitk_window_t *w, xitk_image_t *bg);

void xitk_window_get_window_position (xitk_window_t *w, xitk_rect_t *r);

/* void xitk_window_reparent_window (xitk_window_t *w, xitk_window_t *parent, int x, int y); */
/** title == NULL keeps current. return new title. */
const char *xitk_window_set_window_title (xitk_window_t *w, const char *title);
void xitk_window_set_window_icon (xitk_window_t *w, xitk_image_t *icon);
void xitk_window_set_window_class (xitk_window_t *w, const char *res_name, const char *res_class);

void xitk_window_raise_window(xitk_window_t *w);

void xitk_window_set_transient_for_win(xitk_window_t *w, xitk_window_t *xwin);

/** done_cb status */
#define XITK_WINDOW_DIALOG_BUTTONS_MASK 0x7fff
#define XITK_WINDOW_DIALOG_CHECKED 0x8000
/** preset titles */
typedef enum {
  XITK_MSG_TYPE_INFO = 0,
  XITK_MSG_TYPE_WARN,
  XITK_MSG_TYPE_ERROR,
  XITK_MSG_TYPE_LAST
} xitk_msg_type_t;
#define XITK_MSG_TITLE(_type) ((const char *)(uintptr_t)(_type))
/** button label presets */
#define XITK_LABEL_OK     ((const char *)1)
#define XITK_LABEL_NO     ((const char *)2)
#define XITK_LABEL_YES    ((const char *)3)
#define XITK_LABEL_CANCEL ((const char *)4)
/* non NULL labels show 1-2-3, default preference is 3-1-2. */
xitk_register_key_t xitk_window_dialog_3 (xitk_t *xitk, xitk_window_t *transient_for, int layer_above,
  int width, const char *title,
  void (*done_cb)(void *userdata, int state), void *userdata,
  const char *button1_label, const char *button2_label, const char *button3_label,
  const char *check_label, int checked,
  int text_align, const char *text_fmt, ...) __attribute__ ((format (printf, 14, 15)));

#ifdef YET_UNUSED
void xitk_window_set_modal(xitk_window_t *w);
void xitk_window_dialog_set_modal(xitk_window_t *w);
#endif

/* change WIDGET_TABABLE, WIDGET_FOCUSABLE, WIDGET_CLICKABLE, WIDGET_KEEP_FOCUS, WIDGET_KEYABLE */
int xitk_widget_mode (xitk_widget_t *w, int mask, int mode);

int xitk_clipboard_set_text (xitk_widget_t *w, const char *text, int text_len);

const char *xitk_gettext (const char *text);

#endif
