/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


#include <iostream>

#include <REAL.H>
#include <IntVect.H>
#include <Box.H>
#include <MultiFab.H>
#include <multigrid.H>
#include <ParmParse.H>
#include <globals.H>

int  multigrid::debug     = 0;
int  multigrid::verbose   = 1;
int  multigrid::first_mg  = 1;
Real multigrid::prob_norm = 0.0;

// ************************************************************************
// ** constructor **
// ************************************************************************

multigrid::multigrid(const BoxArray& Grids, 
                     Geometry Geom,
                     MultiFab* Phi,
		     MultiFab* Source, 
                     MultiFab* Resid) :
       grids(Grids), geom(Geom)
{
  if (first_mg) {
      first_mg = 0;
      ParmParse pp("multigrid");
      pp.query("debug",debug);
      pp.query("verbose",verbose);
  }

  // Set pointers to grids.
  phi = Phi;
  source = Source;
  resid = Resid;

  dx = geom.CellSize();

  next = NULL;
}

// ************************************************************************
// ** constructor **
// ************************************************************************

multigrid::multigrid(const BoxArray& Grids, 
                     Geometry Geom,
                     MultiFab* Phi,
		     MultiFab* Source, 
                     MultiFab* Resid, 
                     MultiFab* Area[]) : 
       grids(Grids), geom(Geom)
{
  if (first_mg) {
      first_mg = 0;
      ParmParse pp("multigrid");
      pp.query("debug",debug);
      pp.query("verbose",verbose);
  }

  // Set pointers to grids.
  phi = Phi;
  source = Source;
  resid = Resid;

  dx = geom.CellSize();

  for (int n = 0; n < BL_SPACEDIM; n++)
    area[n] = Area[n];

  next = NULL;
}

// ************************************************************************
// ** solve **
// ************************************************************************

void multigrid::solve(Real tol, 
                      Real Prob_norm,
                      int nngsrb, 
                      int i2)
{
  prob_norm = Prob_norm;

  int i = 1;
  Real rlast = residual();
  Real res = rlast;

  if (rlast > tol) {
    while ((res = vcycle(tol,nngsrb,i2)) > tol) {
      i++;
      if ((verbose == 2 || debug != 0) && ParallelDescriptor::IOProcessor())
        std::cout << "Res/Res0: " << res/prob_norm << " after " 
             << i-1 <<  " iterations " << std::endl;
    }
  }

  if ((verbose >= 1 || debug != 0) && ParallelDescriptor::IOProcessor()) {
    std::cout << "Res/Res0: " << res/prob_norm << " after "
         << i <<  " iterations " << std::endl;
    std::cout << " " << std::endl;
  }
}

// ************************************************************************
// ** vcycle **
// ************************************************************************

Real multigrid::vcycle(Real tol, 
                       int nngsrb, 
                       int i2)
{

  Box prob_domain(grids.minimalBox());

  Real rnorm = residual();
  if (debug && ParallelDescriptor::IOProcessor()) {
    std::cout << "Res/Res0 " << rnorm/prob_norm << " for " << prob_domain.length() <<  std::endl;
  }
  step(nngsrb);
  rnorm = residual();
  if (debug && ParallelDescriptor::IOProcessor()) {
    std::cout << "Res/Res0 " << rnorm/prob_norm << " for " << prob_domain.length() <<  std::endl;
  }

  if (next != NULL) {
    Restrict();
    next->phi->setVal(0.0);
    int downiter = nngsrb;
    int upiter   = i2;
    next->vcycle(tol,downiter,upiter);
    interpolate();
    if (debug) {
      rnorm = residual();
      if (ParallelDescriptor::IOProcessor())
        std::cout << "Res/Res0 " << rnorm/prob_norm << " for " << prob_domain.length() <<  std::endl;
    }
    step(i2);
    if (debug) {
      rnorm = residual();
      if (ParallelDescriptor::IOProcessor())
        std::cout << "Res/Res0 " << rnorm/prob_norm << " for " << prob_domain.length() <<  std::endl;
    }

  } 
    
  return rnorm;
}
