/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* lstat() */
#define _GNU_SOURCE /* asprintf() */
#include <mntent.h> /* interface to /etc/mtab */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <paths.h>  /* _PATH_MOUNTED on Alpine Linux */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>



#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_dll.h"
#include "cdw_logging.h"
#include "canonicalize.h"

/* remember that PATH_MAX already includes ending null:
   1. http://unix.derkeiler.com/Newsgroups/comp.unix.solaris/2005-01/2945.html
   "POSIX (at least IEEE Std 1003.1, 2004 Edition) is clear on this:
        PATH_MAX
	Maximum number of bytes in a pathname, including the terminating
	null character."

   2. http://lkml.indiana.edu/hypermail/linux/kernel/0111.2/1280.html
   "The POSIX.1a draft (the amendment to POSIX.1-1990)
   and XPG4 went with including the null byte in PATH_MAX, and the
   POSIX 1003.1-200x revision (Austin Group) and Single UNIX
   Specification Version 3 also continue this way." */



/* full path to user home directory, should be used by all other modules */
static char *home_dir_fullpath = (char *) NULL;
/* full path to temporary directory, should be used by all other modules */
static char *tmp_dir_fullpath = (char *) NULL;


static cdw_rv_t cdw_fs_init_home_dir_fullpath(void);
static cdw_rv_t cdw_fs_init_tmp_dir_fullpath(void);
static cdw_rv_t cdw_fs_traverse_dir(const char *dirpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);
static cdw_rv_t cdw_fs_traverse_symlink(const char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);



static int cdw_fs_errno_handler_get_index(int e);





/**
   \brief Initialize filesystem module

   Initialize home dir fullpath and tmp dir fullpath. Function first
   attempts to set path to home dir. If this is successful (and only then),
   then the function attempts to set path to tmp dir.

   Function returns CDW_GEN_ERROR if one of the paths can't be set properly
   for any reason.

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_fs_init(void)
{
	cdw_rv_t crv = CDW_OK;
	crv = cdw_fs_init_home_dir_fullpath();
	if (crv != CDW_OK) {
		return CDW_ERROR;
	} else {
		crv =  cdw_fs_init_tmp_dir_fullpath();
		if (crv != CDW_OK) {
			return CDW_ERROR;
		}
	}

	return CDW_OK;
}





/**
   \brief Deallocate resources allocated by fs module

   Function deallocates any resources allocated by cdw_fs_init()
   or other parts of fs module.
*/
void cdw_fs_clean(void)
{
	/* During initialization tmp_dir_fullpath may have been set to
	   be the same pointer as home_dir_fullpath. */
	if (tmp_dir_fullpath != (char *) NULL
	    && (tmp_dir_fullpath != home_dir_fullpath)) {

		free(tmp_dir_fullpath);
		tmp_dir_fullpath = (char *) NULL;
	}

	CDW_STRING_DELETE (home_dir_fullpath);

	return;
}





/**
   \brief Check existence of given file, check also mode/permissions, if required

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Check if given file (regular file or dir) exists, has expected mode/permissions
   and is of given type.

   \p mode is passed to access(2).
   \p mode can be bit sum of symbolic constants: R_OK, W_OK, X_OK.
   \p mode can be also F_OK.

   \param fullpath - full path to given file
   \param mode - file mode
   \param filetype - type of file that you ask about (CDW_FS_FILE or CDW_FS_DIR)

   \return 0 if the path satisfies all constraints
   \return ENOENT if there is no such file as specified by fullpath
   \return EISDIR or ENOTDIR if file exists, but is of wrong kind (FILE / DIR)
   \return EBADF if file is neither dir nor file
   \return lstat()'s errno number if lstat() failed for some reason
   \return access()'s errno number of access() failed
*/
int cdw_fs_check_existing_path(char const * fullpath, int mode, int filetype)
{
	cdw_vdm ("checking path \"%s\"\n", fullpath);
	struct stat finfo;
	if (lstat(fullpath, &finfo) == -1) {
		int e = errno;
		cdw_vdm ("lstat() sets errno '%s' for path '%s'\n", strerror(e), fullpath);
		return e;
	}

	/* check correctness of file type */
	if ( S_ISREG(finfo.st_mode) ) {
		if (filetype == CDW_FS_DIR) {
			return ENOTDIR;
		} else { /*  filetype == CDW_FS_DIR  */
			;
		}
	} else if ( S_ISDIR(finfo.st_mode) ) {
		if (filetype == CDW_FS_FILE) {
			return EISDIR;
		} else { /* filetype == CDW_FS_DIR */
			;
		}
	} else {
		return EBADF;
	}

	/* At this point we know that fullpath points to existing file
	   of expected type, all that can fail now is
	   mode/permissions.  Check mode. */
	int a = access(fullpath, mode);
	if (a == -1) {
		int e = errno;
		cdw_vdm ("access() sets errno '%s' for path '%s'\n", strerror(e), fullpath);
		return e;

	} else { /* (a == 0) */
		cdw_sdm ("access() returns 0 for '%s'\n", fullpath);
		return 0;
	}
}





/**
   \brief Decide whether path is valid

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Check if given \p fullpath meets criteria specified by \p
   expected_* arguments.

   \param fullpath - full path that we want to inspect
   \param expected_file_type - type of file, expected by caller of selection function: CDW_FS_FILE, CDW_FS_DIR
   \param expected_mode - permissions of file, expected by caller of selection function: W_OK, R_OK, X_OK, F_OK (see "man 2 access")
   \param expected_new_or_existing - describes whether selected file has to already exist or not: CDW_FS_NEW, CDW_FS_EXISTING

   \return 0 on success
   \return errno value otherwise
*/
int cdw_fs_check_fullpath(char const * fullpath, int expected_file_type, int expected_mode, int expected_new_or_existing)
{
	cdw_vdm ("INFO: checking path \"%s\"\n", fullpath);
	struct stat finfo;
	int e = cdw_fs_stat(fullpath, &finfo);
	if (e == ENOENT) {
		/* This errno may mean a valid situation: the file
		   doesn't exist, but perhaps we are able and are
		   allowed to create it in parent dir? */
		if (expected_new_or_existing & CDW_FS_NEW) {
			e = cdw_fs_check_access_parent(fullpath, R_OK | W_OK | X_OK);
			if (e == 0) {
				return 0;
			}
		}
		return ENOENT;

	} else if (e != 0) {
		/* All other errno values are real errors. */
		return e;

	} else {
		/* e == 0, file exists, pass to next test. */
	}



	/* File exists in file system. */
	if (expected_new_or_existing == CDW_FS_NEW) {
		/* ... but caller doesn't want existing file. */
		cdw_vdm ("INFO: rejecting existing file, expected only NEW: \"%s\"\n", fullpath);
		return EEXIST;
	} else {
		/* expected_new_or_existing == CDW_FS_NEW | CDW_FS_EXISTING
		   or
		   expected_new_or_existing == CDW_FS_EXISTING */

		/* Existing file is acceptable, but ask for
		   overwriting if mode indicates possibility of
		   writing to the file. */

		if (expected_new_or_existing == CDW_FS_EXISTING && expected_file_type == CDW_FS_DIR) {
			/* There is no way cdw can overwrite existing
			   directory, so returning EEXIST here would
			   not be valid. Continue with checks. */
			;
		} else if (expected_mode == F_OK || expected_mode & W_OK) {
			if (CDW_OK != cdw_fs_errno_handler(EEXIST)) {
				/* Don't overwrite existing file. */
				return EEXIST;
			}
		} else {
			;
		}
	}



	/* Check correctness of file type. We have used stat(), so no
	   need to check if this is link. */
	e = cdw_fs_check_existing_file_type(&finfo, expected_file_type);
	if (e != 0) {
		return e;
	}



	/* At this point we know that fullpath points to existing file
	   of expected type, all that can fail now is mode. Check it. */
	int a = access(fullpath, expected_mode);
	if (a == -1) {
		e = errno;
		cdw_vdm ("INFO: wrong PERMS for existing file: %s: \"%s\"\n", strerror(e), fullpath);
		return e;
	} else { /* (a == 0) */
		cdw_vdm ("INFO: correct PERMS for \"%s\"\n", fullpath);
		return 0;
	}
}





/**
   \brief Call stat(2) on \p fullpath

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Result of call to stat() is returned through \p finfo.

   \param fullpath - path to file to check
   \param finfo - output parameter, pointer to existing struct stat

   \return 0 on stat()'s success
   \return stat()'s errno otherwise
*/
int cdw_fs_stat(char const * fullpath, /* out */ struct stat *finfo)
{
	cdw_vdm ("INFO: stat() on path \"%s\"\n", fullpath);

	if (0 == stat(fullpath, finfo)) {
		cdw_vdm ("INFO: stat() is successful\n");
		return 0;
	}
	int e = errno;

	if (e == ENOENT) {                  /* A component of path does not exist, or path is an empty string. */

		/* File does not exist. This may be a totally
		   acceptable situation, it should be handled
		   separately by caller. */
		cdw_vdm ("INFO: stat() gives ENOENT\n");
		return e;

	} else if (e == EFAULT              /* Bad address. */
		   || e == ENOMEM           /* Out of memory (i.e., kernel memory). */
		   || e == EOVERFLOW        /* (stat()) path refers to a file whose size cannot be represented in the type off_t. */
		   || e == ELOOP            /* Too many symbolic links encountered while traversing the path. */
		   || e == ENOTDIR          /* A component of the path prefix of path is not a directory. */
		   || e == EACCES           /* Search permission is denied for one of the directories in the path prefix of path. */
		   || e == ENAMETOOLONG) {  /* File name too long. */

		/* This is a real error. */
		cdw_vdm ("ERROR: stat() gives error \"%s\"\n", strerror(e));
		return e;

	} else {
		/* This shouldn't happen, all values were covered
		   above. */
		cdw_vdm ("ERROR: stat() gives unexpected error %d / %s\n", e, strerror(e));
		return e;
	}
}





/**
   \brief Check if parent dir exists and has suitable permissions

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Find parent dir of file specified by \p fullpath. Check whether it
   has mode specified by \p expected_mode. Return errno by return
   value.

   The function is intended to check whether it's possible to create
   new files in the parent dir.

   \param fullpath - path to file; check parent dir of this file
   \param expected_mode - mode expected of parent dir

   \return 0 if parent dir has specified mode
   \return ENOENT if path to parent dir can't be constructed
   \return errno of access(2) function otherwise
*/
int cdw_fs_check_access_parent(char const *fullpath, int expected_mode)
{
	char *parent = cdw_fs_shorten_fullpath(fullpath);
	if (!parent) {
		cdw_vdm ("ERROR: can't create parent of fullpath \"%s\"\n", fullpath);
		return ENOENT;
	}

	cdw_vdm ("INFO parent of \"%s\" is \"%s\"\n", fullpath, parent);

	int a = access(parent, expected_mode);
	int e = errno;
	if (a == 0) {
		free(parent);
		parent = (char *) NULL;
		cdw_vdm ("INFO: successfully accessed parent dir \"%s\"\n", parent);
		return 0;
	} else {
		if (e == EACCES) {
			cdw_vdm ("INFO: wrong PERMs of parent dir \"%s\"\n", parent);
		} else if (e == ENOENT || e == ENOTDIR) {
			cdw_vdm ("INFO: parent dir \"%s\" doesn't exist\n", parent);
		} else {
			cdw_vdm ("INFO: sys error when accessing parent dir\"%s\"\n", parent);
		}

		free(parent);
		parent = (char *) NULL;
		return e;
	}
}





/**
   \brief Check whether existing file is of expected file type: regular file or directory

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Based on result of stat(), check whether a file specified by \p
   finfo matches file type specified by \p expected_file_type.

   Function checks only for CDW_FS_FILE or CDW_FS_DIR. If \p
   expected_file_type doesn't contain any of the two, function returns
   error.

   \param finfo - result of stat()
   \param expected_file_type - CDW_FS_FILE | CDW_FS_DIR

   \return ENOTDIR if \p expected_file_type demanded a directory, but \p finfo indicates regular file
   \return EISDIR if \p expected_file_type demanded a regular file, but \p finfo indicates directory
   \return EBADF if \p finfo indicates neither file nor dir
   \return 0 if file indicated by \p finfo meets criteria specified by \p expected_file_type
*/
int cdw_fs_check_existing_file_type(struct stat *finfo, int expected_file_type)
{
	cdw_assert (expected_file_type & CDW_FS_FILE || expected_file_type & CDW_FS_DIR,
		    "ERROR: incorrect value of expected_file_type: 0x%x\n", expected_file_type);

	if ( S_ISREG(finfo->st_mode) ) {
		if (expected_file_type & CDW_FS_FILE) {
			cdw_sdm ("INFO: correct file type for FILE\n");
			return 0;
		} else {
			cdw_vdm ("INFO: rejecting existing file because is FILE, expected DIR\n");
			return ENOTDIR;
		}

	} else if ( S_ISDIR(finfo->st_mode) ) {
		if (expected_file_type & CDW_FS_DIR) {
			cdw_sdm ("INFO: correct file type for DIR\n");
			return 0;
		} else {
			cdw_vdm ("INFO: rejecting existing file because is DIR, expected FILE\n");
			return EISDIR;
		}
	} else {
		cdw_vdm ("WARNING: rejecting file because neither FILE nor DIR\n");
		return EBADF;
	}
}





/**
   \brief Initialize variable with full path to home directory

   Function searches for user's home directory and puts full (non-relative)
   path to the directory into home_dir_fullpath variable.

   First place to search for user's home dir is HOME shell variable. If this
   fails (or in case of other problems) user is asked to enter path to
   home dir manually. If this fails too, or user cancels entering path,
   function returns CDW_ERROR

   \return CDW_OK on success
   \return CDW_ERROR if the variable with path can't be initialized for any reason
*/
cdw_rv_t cdw_fs_init_home_dir_fullpath(void)
{
	cdw_assert (home_dir_fullpath == (char *) NULL, "ERROR: calling init function when path != NULL\n");
	const char *home = getenv("HOME");
	if (home != (char *) NULL) {
		size_t len = strlen(home);
		if (len != 0 && len < PATH_MAX - 1) {
			if (home[len - 1] == '/') {
				cdw_vdm ("WARNING: getenv(\"HOME\") returns path with ending slash\n");
				home_dir_fullpath = strdup(home);
			} else {
				home_dir_fullpath = cdw_string_concat(home, "/", (char *) NULL);
			}
			if (home_dir_fullpath == (char *) NULL) {
				cdw_vdm ("ERROR: failed to initialize home_dir_fullpath with home = \"%s\", len = %zd\n", home, len);
			}
		} else {
			cdw_vdm ("WARNING: strlen(home) out of bounds: len = %zd, PATH_MAX - 1 = %d\n", len, PATH_MAX - 1);
		}
	} else {
		cdw_vdm ("WARNING: getenv(\"HOME\") returns NULL\n");
	}
	/* at this point home_dir_fullpath is either initialized
	   with correct HOME dirpath or is NULL */
	if (home_dir_fullpath == (char *) NULL) {
		/* no sane $HOME, but we need some value in
		   home_dir_fullpath for further function calls */
		home_dir_fullpath = strdup("");
	}

	cdw_vdm ("INFO: $HOME = \"%s\", home_dir_fullpath = \"%s\", strlen(home_dir_fullpath) = %zd\n",
		 home, home_dir_fullpath, strlen(home_dir_fullpath));

	int rv = cdw_fs_check_existing_path(home_dir_fullpath, W_OK, CDW_FS_DIR);
	if (rv == 0) {
		cdw_vdm ("INFO: home directory is initialized as \"%s\" (1)\n", home_dir_fullpath);
		return CDW_OK;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t crv = cdw_file_picker(_("Select directory"),
					       /* 2TRANS: this is message in dialog window. Please keep '\n'. */
					       _("Please select Home directory.\n   Press ESC to cancel."),
					       &home_dir_fullpath,
					       CDW_FS_DIR, W_OK|X_OK, CDW_FS_EXISTING);

		if (crv == CDW_OK) {
			cdw_vdm ("INFO: home directory is initialized as \"%s\" (2)\n", home_dir_fullpath);
			return CDW_OK;
		} else {
			CDW_STRING_DELETE (home_dir_fullpath);

			cdw_vdm ("ERROR: failed to initialize home dir fullpath\n");

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog
					      window, "Home" is user's home dir,
					      "closing" refers to closing
					      application */
					   _("Path to Home directory not initialized properly, closing."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		}
	}
}





/**
   \brief Initialize variable with full path to temporary directory

   Function searches for system temporary directory and puts full (non-relative)
   path to the directory into tmp_dir_fullpath variable.

   First, default value is "/tmp". If this fails user is asked to enter path to
   tmp dir manually. If this fails too, or user cancels entering path,
   function sets tmp_dir_path to home_dir_path - these two pointers have the
   same value.

   \return CDW_OK
*/
cdw_rv_t cdw_fs_init_tmp_dir_fullpath(void)
{
	/* I'm using this assertion here for two reasons:
	   1. if setting home dirpath failed, then setting tmp dirpath
	   will probably fail as well; if setting home dirpath failed
	   and I still call this function to init tmp dirpat, then I made
	   an error somewhere;
	   2. home dir path may be used in this funciton, it is important
	   that home dir is correct */
	cdw_assert (home_dir_fullpath != (char *) NULL,
		    "you wan't to initialize tmp dirpath when home is not initialized (yet)\n");

	/* some sane initial value */
	char *tmp = strdup("/tmp/");
	if (tmp == (char *) NULL) {
		return CDW_ERROR;
	}

	int rv = cdw_fs_check_existing_path(tmp, W_OK, CDW_FS_DIR);
	if (rv == 0) {
		tmp_dir_fullpath = tmp;
		return CDW_OK;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t crv = cdw_file_picker(_("Select directory"),
					       /* 2TRANS: this is message in dialog window. Please keep '\n'. */
					       _("Please select temporary directory.\n   Press ESC to cancel."),
					       &tmp,
					       CDW_FS_DIR, W_OK|X_OK, CDW_FS_EXISTING);

		if (crv == CDW_OK) {
			tmp_dir_fullpath = tmp;
			cdw_sdm ("INFO: tmp directory is initialized as \"%s\"\n", tmp_dir_fullpath);

			return CDW_OK;
		} else {
			CDW_STRING_DELETE (tmp);

			tmp_dir_fullpath = home_dir_fullpath;
			cdw_sdm ("WARNING: initialized tmp dir fullpath as home dir fullpath\n");

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Warning"),
					   /* 2TRANS: this is message in dialog window; */
					   _("Path to tmp directory is the same as path to home directory. This is not a bad thing, but some things might work differently."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			return CDW_OK;
		}
	}
}





/**
   \brief Getter function for tmp_dir_fullpath variable

   Use this function only after initializing fs module (calling
   cdw_fs_init()).

   Returned pointer must not be deallocated.

   \return pointer to string with tmp directory
*/
char const *cdw_fs_get_tmp_dir_fullpath(void)
{
	cdw_assert (tmp_dir_fullpath != (char *) NULL, "did you call cdw_fs_init()?\n");
	size_t len = strlen(tmp_dir_fullpath);
	cdw_assert (len != 0, "len of tmp dir fullpath is 0\n");

	return tmp_dir_fullpath;
}





/**
   \brief Getter function for home_dir_fullpath variable

   Use this function only after initializing fs module (calling
   cdw_fs_init()).

   Returned pointer must not be deallocated.

   \return pointer to string with home directory
*/
char const *cdw_fs_get_home_dir_fullpath(void)
{
	cdw_assert (home_dir_fullpath != (char *) NULL, "did you call cdw_fs_init()?\n");
	size_t len = strlen(home_dir_fullpath);
	cdw_assert (len != 0, "len of home dir fullpath is 0\n");

	return home_dir_fullpath;
}





/**
   \brief Try to get an initial directory path needed for some operations

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   Try to return current working directory.
   If this fails try to return home or tmp directory.
   If this fails try to return root directory.

   Returned pointer is owned by caller.

   \return path to a directory on success
   \return NULL on failure
*/
char *cdw_fs_get_initial_dirpath(void)
{
	char *dirpath = get_current_dir_name();
	if (dirpath) {
		if (CDW_OK == cdw_fs_correct_dir_path_ending(&dirpath)) {
			cdw_vdm ("INFO: initial dirpath is cwd: \"%s\"\n", dirpath);
			return dirpath;
		}
		cdw_vdm ("ERROR: failed to make correct dir path ending for cwd\n");
	}
	cdw_vdm ("ERROR: failed to get cwd, error = \"%s\"\n", strerror(errno));


	dirpath = cdw_fs_get_home_or_tmp_dirpath();
	if (dirpath) {
		cdw_vdm ("INFO: initial dirpath is home or tmp: \"%s\"\n", dirpath);
		return dirpath;
	}
	cdw_vdm ("ERROR: failed to get home or tmp dirpath\n");


	dirpath = strdup("/");
	if (dirpath) {
		cdw_vdm ("INFO: initial dirpath is root: \"%s\"\n", dirpath);
		return dirpath;
	}
	cdw_vdm ("ERROR: failed to strdup() root dir path\n");


	return (char *) NULL;
}





/**
   \brief Return path to either home directory or tmp directory

   Returned pointer is owned by caller.

   \return path to home or tmp directory on success
   \return NULL on failure
*/
char *cdw_fs_get_home_or_tmp_dirpath(void)
{
	char *dirpath = (char *) NULL;


	char const * const home = cdw_fs_get_home_dir_fullpath();
	if (home) {
		dirpath = strdup(home);
		if (dirpath) {
			return dirpath;
		}
		cdw_vdm ("ERROR: failed to strdup() home dir path\n");
	}
	cdw_vdm ("ERROR: failed to get home dir path\n");


	char const * const tmp = cdw_fs_get_tmp_dir_fullpath();
	if (tmp) {
		dirpath = strdup(tmp);
		if (dirpath) {
			return dirpath;
		}
		cdw_vdm ("ERROR: failed to strdup() tmp dir path\n");
	}
	cdw_vdm ("ERROR: failed to get tmp dir path\n");


	return (char *) NULL;
}





/**
   \brief Check if device of given path is mounted in file system

   Use GNU libc interface to /etc/mtab file to determine if given device
   is mounted. Officially you have to pass to a function a string with path
   to device file (like '/dev/xxx') which has to be checked, but
   mount point is also (implicitly) accepted.

   Function should work even if given \p device_fullpath is a symbolic
   link to real device.

   You most probably want to use this function to check if cdrom
   drive is mounted.

   \param device_fullpath - path to device file which has to be checked. No trailing slashes are accepted.

   \return CDW_OK if device is mounted
   \return CDW_NO if device is not mounted
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_fs_check_device_mounted(char const * device_fullpath)
{
	cdw_assert (device_fullpath != (char *) NULL, "device fullpath is null\n");
	struct stat finfo;
	int z = lstat(device_fullpath, &finfo);
	if (z == -1) {
		cdw_vdm ("ERROR: stat() returns !0 (%d) for device_fullpath %s\n", z, device_fullpath);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Path to drive specified in configuration is incorrect. Please check your configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	char *fullpath = (char *) NULL;
	if (!S_ISLNK(finfo.st_mode)) {
		/* device_fullpath is not a link */
		cdw_vdm ("INFO: file specified by fullpath \"%s\" is not a symlink\n", device_fullpath);
		fullpath = strdup(device_fullpath);

	} else { /* S_ISLNK(finfo.st_mode) */
		/* second argument == NULL is standardized in POSIX.1-2008 */
		fullpath = realpath(device_fullpath, (char *) NULL);
		if (fullpath == (char *) NULL) {
			int e = errno;
			cdw_vdm ("ERROR: realpath() can't resolve real path for device_fullpath link %s, strerror = \"%s\"\n", device_fullpath, strerror(e));
			/* 2TRANS: this is message printed in log file,
			   %s is full path to a device file */
			cdw_logging_write(_("cdw can't check device specified in configuration as \"%s\"\n"), device_fullpath);
		}
	}
	if (fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set final fullpath\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("cdw has problems with searching for your device. Please restart cdw."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	FILE *mtab_file = setmntent(_PATH_MOUNTED, "r");
	if (mtab_file == (FILE *) NULL) {
		CDW_STRING_DELETE (fullpath);
		cdw_vdm ("ERROR: call to setmnent() failed\n");
		return CDW_ERROR;
	}

	bool is_mounted = false;

	struct mntent *mnt = (struct mntent *) NULL;
	while ( (mnt = getmntent(mtab_file)) != (struct mntent *) NULL ) {
		int a = strcmp(fullpath, mnt->mnt_fsname);
		int b = strcmp(fullpath, mnt->mnt_dir);

		if (a == 0 || b == 0) {
			is_mounted = true;
			cdw_vdm ("INFO: device %s (%s) is mounted:\n", device_fullpath, fullpath);
			cdw_vdm ("name of mounted file system: %s\n", mnt->mnt_fsname); /* device name, e.g. /dev/scd0 */
			cdw_vdm ("file system path prefix:     %s\n", mnt->mnt_dir);    /* mount point, e.g. /mnt/cdrom (no ending slash) */
			cdw_vdm ("mount type:                  %s\n", mnt->mnt_type);   /* file system type (ext3, iso9660 etc.) */
			cdw_vdm ("mount options:               %s\n\n", mnt->mnt_opts); /* rw,noexec,nosuid,nodev etc. */
			break;
		}
	}

	endmntent(mtab_file);
	mtab_file = (FILE *) NULL;
	/* can't do that, apparently endmntent takes care of this */
	/* free(mnt);
	   mnt = (struct mntent *) NULL; */
	CDW_STRING_DELETE (fullpath);

	if (is_mounted) {
		cdw_vdm ("INFO: device %s is mounted\n", device_fullpath);
		return CDW_OK;
	} else {
		cdw_vdm ("INFO: device %s is not mounted\n", device_fullpath);
		return CDW_NO;
	}
}





/**
   \brief Ensure that given string ends with exactly one '/' char

   Check if given string ends with exactly one '/' char. If there are more
   '/' chars at the end, (without any other chars in between) then function
   removes all of them but one. If there is no ending '/' char, the function
   adds one. Before performing any of these actions, function may modify
   it's argument by removing any white chars from end of it.

   The function can't deal with cases such as "/path/  /", where there are
   two '/' chars at the end, but separated by spaces - function treats this
   as correctly ended string.

   The function can be used to ensure that directory path is properly
   ended with '/'.

   'path' cannot be NULL or empty string. It can be only "/" - function will
   recognize it as correct string.

   If malloc() fails and function returns CDW_ERROR, path is modified
   at most by removing ending white chars.

   Function may realloc 'path'!

   \param path - directory path that you want to check

   \return CDW_OK if success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_fs_correct_dir_path_ending(char **path)
{
	if (*path == NULL) {
		cdw_vdm ("ERROR: path is NULL\n");
		return CDW_ERROR;
	}

	size_t len = strlen(*path);

	if (len == 0) {
		cdw_vdm ("ERROR: path has zero length\n");
		return CDW_ERROR;
	}

#if 0	/* an experiment, don't enable */
	size_t missing_chars = 0;
	bool missing_root = false;
	if (*(*path + 0) != '/'
	    || ((*(*path + 0) != '.') && (*(*path + 1) != '/'))
	    || ((*(*path + 0) != '.') && (*(*path + 1) != '.') && (*(*path + 2) != '/'))) {
		/* neither relative path nor path in root dir */

		cdw_vdm ("INFO: missing root\n");
		missing_root = true;
		missing_chars += 2; /* for two chars: "./" at the beginning of dir path */
	} else {
		cdw_vdm ("INFO: path %s has root\n", *path);
	}
	bool missing_end = false;
	if (file_type == CDW_FS_DIR &&
	    *(*path + len - 1) != '/') {
		cdw_vdm ("INFO: missing end\n");
		missing_end = true;
		missing_chars += 1; /* for one char: "/" at the end of dir path */
	} else {
		cdw_vdm ("INFO: path %s has end\n", *path);
	}

	if (missing_root || missing_end) {
		char *tmp = (char *) malloc(len + missing_chars + 1);
		if (tmp == (char *) NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for new path\n");
			return CDW_ERROR;
		} else {
			size_t offset = 0;
			if (missing_root) {
				snprintf(tmp + offset, 2 + 1, "./");
				offset += 2;
			}
			snprintf(tmp + offset, len + 1 + 1, "%s", *path);
			offset += len;
			if (missing_end) {
				  snprintf(tmp + offset, 1 + 1, "/");
				  offset += 1;
			}
			free(*path);
			*path = tmp;
		  }
	}

	ssize_t i = cdw_fs_get_filename_start(*path);
	cdw_assert (i > 0, "ERROR: can't find last dir name in dirpath \"%s\"\n", *path);
	len = strlen(*path);
	for (; (size_t) i < len; i++) {
		if ((*path)[i] == '/') { /* this is the slash that should end dirpath */
			if ((*path)[i + 1] != '\0') {
				/* and it is not the last char in the string */
				(*path)[i + 1] = '\0';
				break;
			} else {
				/* the slash was already last char in
				   the string, no need to do anything */
				break;
			}
		}
	}

#else

	if (*(*path + len - 1) != '/') {
		char *tmp = realloc(*path, len + 2);
		if (tmp == NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for new path\n");
			return CDW_ERROR;
		}
		*path = tmp;
		*(*path + len) = '/';
		*(*path + len + 1) = '\0';
	} else { /* there is at least one ending '/', but perhaps more */
		ssize_t i = cdw_fs_get_filename_start(*path);
		cdw_sdm ("INFO: path is \"%s\", file name start = %zd\n", *path, i);

		/* the condition is 'i >= 0' because paths can be relative, e.g.
		   "dir3//" - there is no root dir, and i = 0 */

		cdw_assert (i >= 0, "ERROR: can't find last dir name in dirpath \"%s\"\n", *path);

		/* special case, a hack: path is in form of "//" or similar */
		if (i == 1 && (*path)[i] == '/') {
			(*path)[i] = '\0';
			return CDW_OK;
		}

		len = strlen(*path);
		for (; (size_t) i < len; i++) {
			if ((*path)[i] == '/') { /* this is the slash that should end dirpath */
				if ((*path)[i + 1] != '\0') {
					/* and it is not the last char in the string */
					(*path)[i + 1] = '\0';
					break;
				} else {
					/* the slash was already last char in
					   the string, no need to do anything */
					break;
				}
			}
		}
	}
#endif
	return CDW_OK;
}





/**
   \brief Filter function for scandir()

   Filter function for scandir() - returns zero value for current
   directory (".") entry. Entries, for which filter returns zero
   value, are omitted by scandir().

   You needn't worry about passing any argumets to the function - just
   use this function's name as 3rd argument to scandir() call.

   \param entry - item from listing of directory

   \return 0 if entry's file name is ".",
   \return 1 otherwise
*/
int cdw_scandir_filter_one(struct dirent const * entry)
{
	if (!strcmp(entry->d_name, ".")) {
		return 0;
	} else {
		return 1;
	}
}





/**
   \brief Filter function for scandir()

   Filter function for scandir() - returns zero value for current
   directory (".") and parent directory ("..") entries.  Entries, for
   which filter returns zero value, are omitted by scandir()

   You needn't worry about passing any argumets to the function - just
   use this function's name as 3rd argument to scandir() call.

   \param entry - item from listing of directory

   \return 0 if entry's file name is "." or "..",
   \return 1 otherwise
*/
int cdw_scandir_filter_two(struct dirent const * entry)
{
	if (!strcmp(entry->d_name, ".") || !strcmp(entry->d_name, "..")) {
		return 0;
	} else {
		return 1;
	}
}





/**
   \brief Get file of size specified by full path

   TODO: convert returned value type to off_t (used in 'struct stat').

   \return -1 on error
   \return size of file on success
*/
long long cdw_fs_get_file_size(char const * fullpath)
{
	struct stat stbuf;
	int rv = lstat(fullpath, &stbuf);
	if (rv != 0) {
		int e = errno;
		cdw_fs_errno_handler(e);

		return -1;
	} else {
		return (long long) stbuf.st_size;
	}
}





/**
   \brief Copy entries from \p eps to \p list

   \date Function's top-level comment reviewed on 2016-02-21
   \date Function's body reviewed on 2016-02-21

   \p eps is a result of calling scandir() for directory \p dirpath.
   Store information about each file/dir from \p eps on \p list if
   files.

   Caller has to make sure that \p eps does not hold reference to '.' dir.

   \p list is a NULL pointer to doubly-linked list. The pointer, and
   whole list, will be initialized and filled by this function with pointers
   to cdw_file_t data structures.

   \p dirpath must be non-NULL string terminated with '\0', with valid
   path to a directory. This is the same dirpath which was passed to scandir()
   to get \p eps.

   \p eps must be result of call of scandir() with given \p dirpath

   \p n_files is a number of file names stored in \p eps, a value returned
   by scandir() when called with given \dirpath. Make sure that \p n_files is
   non-zero, therefore \p eps has to have at least one file name. It can't be
   "." - current dir, but it can be ".." - parent dir.

   \p list won't include hidden files if \p include_hidden is false.

   \p list will be sorted by type of entries: first there will be
   directories, then there will be files.

   \param list - will be set to list of files in given dir \p dirpath
   \param dirpath - path to directory that you scan
   \param eps - list of file names in given dir \p dirpath
   \param n_files - number of file names on \p eps
   \param include_hidden - controls if hidden UNIX files should be copied from \p eps to \p list

   \return number of items copied to the list on success (there always should be at least one: "..")
   \return 0 on errors
*/
size_t cdw_fs_copy_dirent_to_list(cdw_dll_item_t **list, char const * dirpath, struct dirent **eps, size_t n_items, bool include_hidden)
{
	cdw_assert (*list == (cdw_dll_item_t *) NULL, "ERROR: can't use non-NULL list\n");
	cdw_assert (dirpath, "ERROR: can't use NULL dirpath\n");
	cdw_assert (n_items > 0, "ERROR: you want to create list of zero files\n");

	/* List heads will be allocated by cdw_dll_append(). */
	cdw_dll_item_t *dirs = (cdw_dll_item_t *) NULL;    /* List of dirs copied from eps. */
	cdw_dll_item_t *files = (cdw_dll_item_t *) NULL;   /* List of files copied from eps. */

	size_t n_dirs = 0;
	size_t n_files = 0;
	size_t n_hidden = 0;
	bool success = true;

	/* Explicitly place ".." dir on the list as first item.

	   Placing it as first on the list will ensure that it's
	   displayed as first, even if other items are alphabetically
	   before it.

	   Placing it explicitly will ensure that reference to parent
	   dir is visible even if hidden items should not be
	   displayed.  Parent dir must not be affected by this option.

	   In every dir displayed by cdw there will be always a '..'
	   dir, even in root directory. */
	cdw_file_t *parent = cdw_file_new(dirpath, "..");
	if (!parent) {
		cdw_vdm ("ERROR: failed to create parent file %s/%s\n", dirpath, "..");
		return 0;
	}
	cdw_dll_append(&dirs, (void *) parent, cdw_file_equal);
	n_dirs++;

	for (size_t i = 0; i < n_items; i++) {
		/* Skip parent dir, regardless of 'include_hidden'.
		   The dir is already on the list. Notice that it's
		   not included in n_hidden. */
		if (eps[i]->d_name[0] == '.' && eps[i]->d_name[1] == '.' && eps[i]->d_name[2] == '\0') {
			continue;
		}

		if (cdw_fs_is_hidden(eps[i]->d_name) && !include_hidden) {
			n_hidden++;
			continue;
		}

		cdw_file_t *file = cdw_file_new(dirpath, eps[i]->d_name);
		if (!file) {
			cdw_vdm ("ERROR: failed to create file for \"%s\"/\"%s\"\n", dirpath, eps[i]->d_name);
			success = false;
			break;
		}
		if (file->type == CDW_FS_DIR) { /* TODO: shouldn't it be "file->type & CDW_FS_DIR"? */
			cdw_dll_append(&dirs, (void *) file, cdw_file_equal);
			n_dirs++;
		} else { /* CDW_FS_FILE or CDW_FS_OTHER */
			cdw_dll_append(&files, (void *) file, cdw_file_equal);
			n_files++;
		}
	}

	if (success) {
		/* connect 'dirs' list and 'files' list into one */

		cdw_assert (n_dirs + n_files + n_hidden == n_items, "ERROR: numbers of dirs/files doesn't add up\n");
		/* There will always be at least ".." dir. */
		cdw_assert (n_dirs > 0, "ERROR: zero directories in path \"%s\"\n", dirpath);

		/* There always are some dirs (at least one of them:
		   "..").  Output list starts with directories,
		   followed by files. */
		*list = dirs;
		if (n_files && n_dirs) {
			cdw_assert (files, "ERROR: counted %zd files, but list of files is NULL\n", n_files);
			cdw_dll_item_t *last_dir = cdw_dll_ith_item(dirs, n_dirs - 1);
			last_dir->next = files;
			files->prev = dirs;
		}
		return n_files + n_dirs;
	} else {
		if (dirs) {
			cdw_file_dealloc_files_from_list(dirs);
			cdw_dll_clean(dirs);
			dirs = (cdw_dll_item_t *) NULL;
		}
		if (files) {
			cdw_file_dealloc_files_from_list(files);
			cdw_dll_clean(files);
			dirs = (cdw_dll_item_t *) NULL;
		}
		return 0;
	}
}




/**
   \brief Get position in fullpath where file name starts

   Given a path \p fullpath, find the position (start) of last
   component of the path.

   So for "/home/user/.emacs" the function should return 11.

   \param fullpath - path to be examined

   \return start of file name in fullpath on success
   \return -1 otherwise
*/
ssize_t cdw_fs_get_filename_start(char const * fullpath)
{
	if (fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: argument is NULL\n");
		return -1;
	}
	char *path = strdup(fullpath);
	cdw_assert (path != (char *) NULL, "ERROR: failed to strdup() input string\n");

	size_t len = strlen(path);
	if (len == 0) {
		free(path);
		path = (char *) NULL;
		cdw_vdm ("ERROR: argument has length = 0\n");
		return -1;
	}

	/* put "end" at last char in string that is not '/' */
	ssize_t end = (ssize_t) len - 1;
	while (end > 0 && path[end] == '/') {
		end--;
	}

	if (end == 0) {
		if (path[0] == '/') {
			/* path is "/" or "///" or "// // " or sth like that,
			   just slashes and spaces; in either case file name
			   start is at position 1 */
			end = 1;
		} else {
			/* path is "a/" or "a " */
			end = 0;
		}
	} else {
		/* "end" is now at the end of path, but before ending '/';
		   search for '/' that stands before last name in fullpath */
		while (end >= 0 && path[end] != '/') {
			end--;
		}
		end++;
	}

	free(path);
	path = (char *) NULL;
	return end;
}





/**
   \brief Top-level function for visiting a directory, its entries, and its subdirectories

   \param dirpath - full path to directory that you want to traverse
   \param visitor - function that will operate on all entries in the directory and its subdirs
   \param visitor_data - (not completely) private data of visitor function

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_fs_traverse_path(char const * fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	cdw_vdm ("INFO: TRAVERSE PATH: \"%s\"\n", fullpath);

	/* WARNING: don't use stat() - it yelds bad results for symlinks;
	   at this point we don't want to follow symbolic links, we only
	   need to know if file pointed by path is a symlink or not;
	   we will examine symlinks more closely below */
	struct stat finfo;
	int ok = lstat(fullpath, &finfo);
	int e = errno;
	if (ok != 0) {
		/* TODO: that is weird, but lstat() fails on broken links;
		   this is not a bad thing in this context that we return
		   here, but I thought that lstat won't fail on broken links */
		/* TODO: perhaps we could write skipped paths to log file
		   without returning? */
		cdw_vdm ("ERROR: lstat fails for \"%s\", err = \"%s\"\n", fullpath, strerror(e));
		/* 2TRANS: this is message printed to log file; %s is
		   a full path to file */
		cdw_logging_write(_("Error: cdw can't get any information about this file: \"%s\"\n"), fullpath);
		/* 2TRANS: this is message printed to log file; %s is
		   an additional error message string */
		cdw_logging_write(_("Error: perhaps this information can help resolve problem: \"%s\"\n"), strerror(e));
		// assert (0);
		return CDW_ERROR;
	}

	cdw_rv_t retval = CDW_NO;

	if ( S_ISREG(finfo.st_mode) ) { /* regular file */
		cdw_vdm ("INFO: regular file (%s)\n", fullpath);
		visitor(fullpath, &finfo, visitor_data);
		retval = CDW_OK;
	} else if ( S_ISLNK(finfo.st_mode) ) { /* symbolic link */
		cdw_vdm ("INFO: symbolic link (%s)\n", fullpath);
		if (visitor_data->follow_symlinks) {
			cdw_vdm ("INFO: following symbolic link\n");
			retval = cdw_fs_traverse_symlink(fullpath, visitor, visitor_data);
		} else {
			cdw_vdm ("INFO: NOT following symbolic link\n");
			visitor(fullpath, &finfo, visitor_data);
			retval = CDW_OK;
		}
	} else if ( S_ISDIR(finfo.st_mode) ) { /* directory */
		cdw_vdm ("INFO: directory (%s)\n", fullpath);
		visitor(fullpath, &finfo, visitor_data);
		retval = cdw_fs_traverse_dir(fullpath, visitor, visitor_data);
	} else { /* other, non-appendable file type */
		cdw_vdm ("INFO: other file (%s)\n", fullpath);
		retval = CDW_NO;
	}

	if (retval == CDW_ERROR) {
		/* 2TRANS: this is message printed to log file;
		   %s is a full path to file */
		cdw_logging_write(_("Error: cdw can't get any information about this file: \"%s\"\n"), fullpath);
		if (e != 0) {
			/* 2TRANS: this is message printed to log file; %s is
			   an additional error message string */
			cdw_logging_write(_("Error: perhaps this information can help resolve problem: \"%s\"\n"), strerror(e));
		}
	}

	return retval;
}





cdw_rv_t cdw_fs_traverse_dir(const char *dirpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	struct dirent **eps = (struct dirent **) NULL;
	/* get directory listing - this allocates memory for **eps */
	int n = scandir(dirpath, &eps, cdw_scandir_filter_two, &alphasort);
	int e = errno;
	/* scandir(3): "The scandir() function returns the number of
	   directory entries selected or -1 if an error occurs." */
	cdw_rv_t retval = CDW_OK;
	if (n == -1) {
		cdw_vdm ("ERROR: scandir() returns -1 (%s) for dirpath \"%s\"\n", strerror(e), dirpath);
	       retval = CDW_ERROR;
	} else if (n == 0) {
		/* TODO: empty dir, is there anything more to do? */
		retval = CDW_OK;
	} else {
		for (int i = 0; i < n; i++) {
			/* for every dir entry create new full path to this entry and scan it */
			char *newpath = (char *) NULL;
			/* TODO: why use '/' here if dir paths should
			   end with '/' ? */
			int rv = asprintf(&newpath, "%s/%s", dirpath, eps[i]->d_name);
			if (rv == -1) {
				cdw_vdm ("ERROR: failed to allocate memory for new path\n");
				retval = CDW_ERROR;
				break;
			} else {
				cdw_rv_t crv = cdw_fs_traverse_path(newpath, visitor, visitor_data);
				free(newpath);
				newpath = (char *) NULL;
				if (crv != CDW_OK) {
					cdw_vdm ("ERROR: function failed to traverse path \"%s\"\n", newpath);
					retval = CDW_ERROR;
					break;
				}
			}
		}

	}

	if (eps != (struct dirent **) NULL) {
		for (int i = 0; i < n; i++) {
			free(eps[i]);
			eps[i] = (struct dirent *) NULL;
		}
		free(eps);
		eps = (struct dirent **) NULL;
	}

	return retval;
}





cdw_rv_t cdw_fs_traverse_symlink(const char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	/* CAN_EXISTING - all elements of a path must exist */
	char *resolved_fullpath = canonicalize_filename_mode(fullpath, CAN_EXISTING);
	int e = errno;
	if (!resolved_fullpath) {
		cdw_vdm ("ERROR: failed to fetch target for link \"%s\"\n", fullpath);
		cdw_vdm ("ERROR: error = \"%s\"\n", strerror(e));

#if 0 /* enable after checking behavior of stat() and lstat() on broken links */
		if (errno == ENOENT) {

			struct stat link_info;
			int sym1 = lstat(new_fullpath, &link_info);
			int sym2 = stat(new_fullpath, &link_info);
			if (sym1 == 0 && sym2 != 0) {
				cdw_vdm ("  errno == ENOENT && the link is broken\n");
				/* we can broken link and happily scan dir further */
			} else {
				cdw_vdm ("  errno == ENOENT, but reason is other than broken link\n");
				/* TODO: some more information for user pls */
				return CDW_ERROR;
			}
		}
#endif

		return CDW_ERROR;
	} else {
		cdw_rv_t retval = CDW_OK;
		if (!strcmp(resolved_fullpath, ".")) {
			cdw_vdm ("INFO: link \"%s\" resolved as \".\", skipping\n", fullpath);
			retval = CDW_OK;
		} else {
			cdw_vdm ("INFO: link \"%s\" resolved as new fullpath = \"%s\", checking it...\n",
				 fullpath, resolved_fullpath);
			retval = cdw_fs_traverse_path(resolved_fullpath, visitor, visitor_data);
		}

		CDW_STRING_DELETE (resolved_fullpath);
		return retval;
	}
}





/**
   \brief Visitor function that collects size of all visited files

   A function that traverses directory tree should call this function
   for each file that is present in that directory tree.

   \param fullpath - path to visited directory. For debug purposes only.
   \param finfo - result of lstat() on \p fullpath
   \param data - visitor's data, where files' size is accumulated; also contains some flags
 */
void cdw_fs_visitor(char const * fullpath, struct stat const * finfo, cdw_fs_visitor_data_t *data)
{
	if ( S_ISREG(finfo->st_mode) ) { /* regular file */
		/* st_size is file size, in bytes */
		cdw_vdm ("INFO: file, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
		data->size += (long long) finfo->st_size;

		if (finfo->st_size >= ((off_t) 4 * CDW_1_GIGA)) { /* 4 GB, TODO: ">" or ">=" ?? */
			cdw_vdm ("WARNING: file larger than 4 GB\n");
			data->has_file_over_4GB = true;
		}
	} else if ( S_ISLNK(finfo->st_mode) ) { /* symbolic link */
		if (data->follow_symlinks) {
			/* Visitor is not meant to follow links, just
			   calculate sizes. Following symlinks is a
			   job of a function that traverses dirs. */
		} else {
			cdw_vdm ("INFO: link, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
			/* st_size is length of path without terminating null */
			data->size += finfo->st_size;

		}
	} else if ( S_ISDIR(finfo->st_mode) ) { /* directory */
		cdw_vdm ("INFO:  dir, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
		/* experimental results: using "finfo->st_blocks * finfo->st_blksize"
		   gives results closer to reality than "finfo->st_size" */
		/* FIXME: don't guess why it works, KNOW why it works */
		data->size += (finfo->st_blocks * finfo->st_blksize);
	} else {
		;
	}

#ifndef NDEBUG
	if (data->size < 1024) {
		cdw_vdm ("INFO: accumulated data->size = %lld B\n", data->size);
	} else if (data->size < CDW_1_MEGA) {
		cdw_vdm ("INFO: accumulated data->size = %.2f kB\n", (double) data->size / 1024.0);
	} else {
		cdw_vdm ("INFO: accumulated data->size = %.2f MB\n", (double) data->size / (1.0 * CDW_1_MEGA));
	}
#endif

	return;
}







/* 2TRANS: this is title of dialog window */
const char *title_error = "Error";
/* 2TRANS: this is title of dialog window */
const char *title_warning = "Warning";

struct {
	int e;
	const char *title;
	const char *message;
	int dialog_type;
	int colors;
	int crv;
} cdw_fs_errno_table[] = {
	{ EEXIST, /* pathname already exists and O_CREAT and O_EXCL were used */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Warning"),
	  /* 2TRANS: this is message dialog window, user can
	     press Yes, No or Cancel button */
	  gettext_noop("File already exists. Overwrite?"),
	  /* 0 - special case, return value will be taken from cdw_buttons_dialog */
	  CDW_BUTTONS_YES_NO_CANCEL, CDW_COLORS_WARNING, 0 },

	{ EISDIR, /* pathname refers to a directory and the access requested
		     involved writing (that is, O_WRONLY or O_RDWR is set) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("The path points to directory."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EACCES, /* The requested access to the file is not allowed, or
		     search permission is denied for one of the directories
		     in the path prefix of pathname, or the file did not
		     exist yet and write access to the parent directory is
		     not allowed */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Wrong access rights to file or parent directory."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENAMETOOLONG, /* pathname was too long */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is too long."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOENT, /* O_CREAT is not set and the named file does not exist.
		     Or, a directory component in pathname does not exist
		     or is a dangling symbolic link */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File or part of directory path does not exist."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EBADF,
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is of incorrect type."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOTDIR, /* A component used as a directory in pathname is not,
		      in fact, a directory, or O_DIRECTORY was specified and
		      pathname was not a directory */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("This is not path to directory or part of a path name is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EPERM, /* The O_NOATIME flag was specified, but the effective user
		    ID of the caller did not match the  owner of the file
		    and the caller was not privileged (CAP_FOWNER) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Wrong permissions for the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENXIO, /* O_NONBLOCK | O_WRONLY is set, the named file is a FIFO
		    and no process has the file open for reading. Or, the
		    file is a device special file and no corresponding
		    device exists */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid or points to invalid file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENODEV, /* pathname refers to a device special file and no
		     corresponding device exists. (This is a Linux kernel bug;
		     in this situation ENXIO must be returned.) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid or points to invalid file"),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EROFS, /* pathname refers to a file on a read-only filesystem and
		    write access was requested */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path to file in read-only file system."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ETXTBSY, /* pathname refers to an executable image which is currently
		      being executed and write access was requested */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("The file pointed by path is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EFAULT, /* pathname points outside your accessible address space */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EFBIG, /* pathname  refers to a regular file, too large to be opened */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is too big to open."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EOVERFLOW, /* pathname  refers to a regular file, too large to be opened (POSIX.1-2001) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is too big to open."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ELOOP, /* Too many symbolic links were encountered in resolving
		    pathname, or O_NOFOLLOW was specified but pathname was
		    a symbolic link */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path uses too many symbolic links."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOSPC, /* pathname was to be created but the device containing
		     pathname has no room for the new file */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("There is no space on this device for the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOMEM, /* Insufficient kernel memory was available */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("No system memory. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ EMFILE, /* The process already has the maximum number of files open */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Too many files opened. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ ENFILE, /* The system limit on the total number of open files has been reached */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Too many files opened. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ EWOULDBLOCK, /* The O_NONBLOCK flag was specified,and an incompatible lease
			  was held on the file (see fcntl(2)) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Unknown error while accessing the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ 0, /* arbitrary value in guard element */
	  (char *) NULL, /* guard value */
	  (char *) NULL, /* guard value */
	  0, 0, 0 /* arbitrary values in guard element */ }};



int cdw_fs_errno_handler_get_index(int e)
{
	int i = 0;
	while (cdw_fs_errno_table[i].title != (char *) NULL) {
		if (cdw_fs_errno_table[i].e == e) {
			return i;
		}
		i++;
	}
	return -1;
}



/**
   \brief Wrapper for code checking errno value and informing user about results

   This code checks value of errno and puts dialog window informing user
   about current situation. The function doesn't take any actions to fix
   a problem that caused some function to set errno. Caller must take care
   of this.

   \param e - errno value to examine

   \return CDW_SYS_ERROR if \p e signifies some system error
   \return CDW_NO if path is invalid (because of some non-critical error)
           or if file already exists, but user don't want to overwrite it
           (caller should ask for another path)
   \return CDW_CANCEL  if file already exists and user don't want to
           overwrite nor enter new path
   \return CDW_OK if file already exists and user wants to overwrite it;
           caller has to check the file permissions before attempting to
           overwrite;
*/
cdw_rv_t cdw_fs_errno_handler(int e)
{
	int i = cdw_fs_errno_handler_get_index(e);
	if (i != -1) {
		cdw_rv_t crv = cdw_buttons_dialog(cdw_fs_errno_table[i].title,
						  cdw_fs_errno_table[i].message,
						  cdw_fs_errno_table[i].dialog_type,
						  cdw_fs_errno_table[i].colors);
		if (e == EEXIST) {
			/* EEXIST - pathname already exists and O_CREAT and O_EXCL were used */
			return crv;
		} else {
			return cdw_fs_errno_table[i].crv;
		}
	} else {
		/* "e" was not found in errors table */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("System error"),
				   /* 2TRANS: this is message in
				      dialog window, user can
				      press OK button */
				   _("Unknown error while accessing the file."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}
}





bool cdw_fs_is_hidden(char const * file_name)
{
	return file_name[0] == '.';
}





/**
   \brief Remove last part of a path

   Create a new fullpath by removing last part (file name or directory
   name) from fullpath.

   In contrast to all other places, dirpath returned by this function
   does not end with a slash. This is a special case needed in special
   place.

   In extreme cases returned string may be empty (containing only
   NUL), but that really shouldn't happen for canonical fullpath.

   \p path must be a canonical path.

   Returned pointer is owned by caller.

   \param path - initial value of path

   \return new, shortened path on success
   \return NULL on failure of strdup()
*/
char *cdw_fs_shorten_fullpath(char const * fullpath)
{
	char *dirpath = strdup(fullpath);
	if (!dirpath) {
		return (char *) NULL;
	}

	char *slash = strrchr(dirpath, '/');

	if (!slash) {
		/* No slashes. Treat whole string as file name and
		   remove it from returned string. */
		dirpath[0] = '\0';
		return dirpath;
	}

	/* Remove a possible ending slash from dir path. */
	if (slash != dirpath
	    && *(slash + 1) == '\0') {

		*(slash) = '\0';
		slash = strrchr(dirpath, '/');
	}


	if (slash == dirpath) {
		/* A file in root directory. Return only root directory. */
		dirpath[1] = '\0';
		return dirpath;
	}

	*slash = '\0';

	cdw_vdm ("INFO: original path: \"%s\", shortened path: \"%s\"\n", fullpath, dirpath);

	return dirpath;
}





/**
   \brief Check if given fullpath is a symlink, and return target path for it

   This is a simple version of canonicalize_filename_mode(), only
   resolving symbolic links. It's basically a wrapper around readlink().

   \param fullpath - fullpath to inspect/resolve

   \return freshly allocated buffer with resolved path if \p fullpath is a symbolic link
   \return NULL pointer on errors or when \p fullpath is not a symbolic link
*/
char *cdw_fs_get_target_path_simple(char const * fullpath)
{
	char *target_path = (char *) NULL;

	char link_path[PATH_MAX - 2];
	ssize_t r = readlink(fullpath, link_path, PATH_MAX - 2);
	if (r != -1) {
		link_path[r] = '\0';

		target_path = strdup(link_path);
		cdw_sdm ("INFO: \"%s\" -> \"%s\"\n", fullpath, target_path);
	} else if (r == EINVAL) {
		/* "The named file is not a symbolic link." */
	} else {
		; /* TODO: check rest of values of r, some of them might
		     indicate serious problems */
	}

	return target_path;
}










// #define CDW_UNIT_TEST_CODE /* definition used during development of unit tests code */
#ifdef CDW_UNIT_TEST_CODE

/* *** unit test functions *** */




static void test_cdw_fs_correct_dir_path_ending(void);
static void test_cdw_fs_get_filename_start(void);
static void test_gnulib_canonicalize_filename_mode(void);
static void test_cdw_fs_is_hidden(void);
static void test_cdw_fs_shorten_fullpath(void);



void cdw_fs_run_tests(void)
{
	fprintf(stderr, "testing cdw_fs.c\n");

	/* cdw_fs_correct_dir_path_ending() depends on cdw_fs_get_filename_start(),
	   test cdw_fs_get_filename_start() first */
	test_cdw_fs_get_filename_start();
	test_cdw_fs_correct_dir_path_ending();
	test_gnulib_canonicalize_filename_mode();
	test_cdw_fs_is_hidden();
	test_cdw_fs_shorten_fullpath();

	fprintf(stderr, "done\n\n");

	return;
}





/* I'm sure that canonicalize_filename_mode() works just fine,
   but I want to test how exactly it behaves;

   Note that only CAN_MISSING has been used */
void test_gnulib_canonicalize_filename_mode(void)
{
	fprintf(stderr, "\ttesting cdw_fs_canonicalize_path()...");

	struct {
		int type;
		const char *input;
		const char *expected_result;
	} test_data[] = {
		/* test with only root dir */
		{ CAN_MISSING, "/", "/"},
		/* test normal situation, where path is not modified */
		{ CAN_MISSING, "/my/simple/test/dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out "/./" elements */
		{ CAN_MISSING, "/my/./simple/test/dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out mix of "/./" and "//" elements */
		{ CAN_MISSING, "/my///simple/////test/././//dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out mix of more "/./" and "//" elements */
		{ CAN_MISSING, "/.//./././/my////.//simple/./test/dirs/string", "/my/simple/test/dirs/string"},
		/* test behavior with reference to parent dir - simple case */
		{ CAN_MISSING, "/my/simple/../test/dirs/string", "/my/test/dirs/string"},
		/* test behavior with reference to parent dir - more complicated case */
		{ CAN_MISSING, "/../../simple/../../test/dirs/string", "/test/dirs/string"},
		/* test behavior with mix of "/../", "/./" and "//" */
		{ CAN_MISSING, "/..///../base/dir///.././/of/.//./my/simple/../././//././/.//..//test/dirs/string", "/base/of/test/dirs/string"},
		/* "...", "..of..", "..my." are valid dir names, not special entities */
		{ CAN_MISSING, "/base/.../..of..//..my./simple/...//./..//././/.//..//test/dirs../string.../////// ", "/base/.../..of../..my./test/dirs../string.../ "},
		/* test removing "." from end of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string/.", "/my/simple/test/dirs/string"},
		/* test removing ending ".." and last dir from of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string/..", "/my/simple/test/dirs"},
		/* test removing ending ".." and last dir from of path - second case */
		{ CAN_MISSING, "/my/simple/test/dirs/string/../..//", "/my/simple/test"},
		/* test removing ending ".."/"." and last dir from of path - third case */
		{ CAN_MISSING, "/my/simple/test/dirs/string/..//../.", "/my/simple/test"},
		/* test removing duplicated "/" from end of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string//", "/my/simple/test/dirs/string"},
		{ CAN_MISSING, (char *) NULL, (char *) NULL }
	};

	int i = 0;
	while (test_data[i].input != (char *) NULL) {
		char *result = canonicalize_filename_mode(test_data[i].input, test_data[i].type);
		cdw_assert_test (!strcmp(result, test_data[i].expected_result),
				 "ERROR: failed test #%d: \"%s\"\n", i, result);
		free(result);
		result = (char *) NULL;
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_fs_correct_dir_path_ending(void)
{
	fprintf(stderr, "\ttesting cdw_fs_correct_dir_path_ending()... ");

	cdw_rv_t crv = CDW_OK;
	char *path = (char *) NULL;

	/* first test basic cases */

	crv = cdw_fs_correct_dir_path_ending(&path);
	assert(crv == CDW_ERROR);

	crv = cdw_string_set(&path, "");
	assert(crv == CDW_OK);
	crv = cdw_fs_correct_dir_path_ending(&path);
	assert(crv == CDW_ERROR);

	struct {
		const char *input;
		const char *output;
	} test_data[] = {
		/* some cases with more or less correct input */
		{ "/home",           "/home/" },
		{ "/home ",          "/home /" },
		{ "/home/",          "/home/" },
		{ "/home/ ",         "/home/ /" },

		/* the function can handle (i.e. remove) any number of slashes */
		{ "/home///////",    "/home/" },

		/* some tests with short strings */
		{ "/ho///",          "/ho/" },
		{ "h/////",          "h/" },
		{ "// ///",          "// /" },
		{ "/",               "/" },
		{ "/ ",              "/ /" },

		{ (char *) NULL,     (char *) NULL }
	};

	int i = 0;
	while (test_data[i].input != (char *) NULL) {

		crv = cdw_string_set(&path, test_data[i].input);
		cdw_assert_test (crv == CDW_OK, "ERROR: failed test #%d\n", i);
		crv = cdw_fs_correct_dir_path_ending(&path);
		cdw_assert_test (crv == CDW_OK, "ERROR: failed test #%d\n", i);
		int d = strcmp(path, test_data[i].output);
		cdw_assert_test (d == 0, "ERROR: test #%d failed, path should be \"%s\", is \"%s\"\n",
				 i, test_data[i].output, path);

		i++;
	}

	free(path);
	path = (char *) NULL;

	fprintf(stderr, "OK\n");
	return;

}





void test_cdw_fs_get_filename_start(void)
{
	fprintf(stderr, "\ttesting cdw_fs_get_filename_start()... ");

	struct {
		int expected_result;
		const char *path;
	} test_data[] = {

		/* first test simple cases without root dir */
		{ -1,  "" },
		{  0,  "n" },
		{  0,  "name" },
		{  0,  "name " },
		{  0,  "na me " },
		{  0,  " name" },

		/* more complicated cases without root dir */

		{  0,  "name/" },
		{  0,  "n/" },
		{  6,  "name / " },
		{  7,  "na me / " },
		{  4,  " na/me//" },
		{  6,  "my na/me of file////////" },

		/* test simple cases with root dir */

		{  1,  "/" },
		{  1,  "/a" },
		{  1,  "/file name /" },
		/* path is not canonicalized (i.e. leading "//" aren't compressed
		   into one), so file name starts at 2 */
		{  2,  "//file name /" },
		{  12,  "/file name / other /////" },

		/* test few unusual cases */

		{  1,  "/////" },
		/* TODO: is this a correct test case at all? */
		{  8,  " / / / / " },
		{  0,  "     " },

		{  0,  (char *) NULL }
	};

	int i = 0;
	while (test_data[i].path != (char *) NULL) {
		ssize_t pos = cdw_fs_get_filename_start(test_data[i].path);
		cdw_assert_test (pos == test_data[i].expected_result,
				 "ERROR: test #%d failed: input path = \"%s\", pos = %zd, expected %d\n",
				 i, test_data[i].path, pos, test_data[i].expected_result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_fs_is_hidden(void)
{
	fprintf(stderr, "\ttesting cdw_fs_is_hidden()... ");

	struct {
		const char *filename;
		bool expected_result;
	} test_data[] = {

		/* first test simple cases without root dir */
		{ "",             false },
		{ ".",            true  }, /* Reference to "this dir". */
		{ "..",           true  }, /* Reference to "parent dir". */
		{ "...",          true  }, /* File named "...". */
		{ ".hidden",      true  },
		{ "..hidden",     true  },
		{ "not hidden",   false },
		{ " not hidden",  false },

		{ (char *) NULL, false  }
	};

	int i = 0;
	while (test_data[i].filename) {
		bool hidden = cdw_fs_is_hidden(test_data[i].filename);
		cdw_assert_test (hidden == test_data[i].expected_result,
				 "ERROR: test #%d failed: input file name = \"%s\", result = %d, expected %d\n",
				 i, test_data[i].filename, hidden, test_data[i].expected_result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_fs_shorten_fullpath(void)
{
	fprintf(stderr, "\ttesting cdw_fs_shorten_fullpath()... ");

	struct {
		char const * fullpath;
		char const * expected_shortened;

	} test_data[] = {

		{ "/home/acerion/one/",      "/home/acerion" },
		{ "/home/acerion/two",       "/home/acerion" },
		/* shorten_fullpath() is not canonicalization
		   function, so it doesn't interpret ".." before
		   shortenting the path. It's just a dummy string
		   processing function. */
		{ "/home/acerion/../",       "/home/acerion" },
		{ "/home/acerion/..",        "/home/acerion" },


		{ "/home/",                  "/"      },
		{ "/home",                   "/"      },
		{ "/",                       "/"      },

		{ "home/acerion/",           "home"   },
		{ "home/acerion",            "home"   },
		{ "home/../",                "home"   },
		{ "home/..",                 "home"   },

		{ "home",                    ""       },
		{ "home",                    ""       },
		{ "",                        ""       },

		{ (char *) NULL,  (char *) NULL }
	};

	int i = 0;
	while (test_data[i].fullpath) {

		char * shortened = cdw_fs_shorten_fullpath(test_data[i].fullpath);
		cdw_assert_test (0 == strcmp(shortened, test_data[i].expected_shortened),
				 "ERROR: test #%d failed: input fullpath = \"%s\", result = \"%s\", expected \"%s\"\n",
				 i, test_data[i].fullpath, shortened, test_data[i].expected_shortened);

		CDW_STRING_DELETE (shortened);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}





#endif /* CDW_UNIT_TEST_CODE */
