/*-
 * Copyright (c) 2008  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Ringlet$
 */

#define _GNU_SOURCE

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "confget.h"
#include "confget_common.h"

/* Is conffile pointing to stdin? */
static int	 is_stdin;

/***
 * Function:
 *	common_openfile		- open any type of file for reading
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Stores the opened file into conffile.
 */
int
common_openfile(void)
{
	FILE *ifp;

	if (filename == NULL) {
		warnx("No configuration file name specified");
		return (-1);
	}
	if (!strcmp(filename, "-")) {
		conffile = stdin;
		is_stdin = 1;
		return (0);
	}

	ifp = fopen(filename, "r");
	if (ifp == NULL) {
		warn("Opening the input file %s", filename);
		return (-1);
	}
	conffile = ifp;
	is_stdin = 0;
	return (0);
}

/***
 * Function:
 *	common_openfile_null	- do not open any file for methods that do not
 *				  take their input from files
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Stores a null value into conffile.
 */
int
common_openfile_null(void)
{

	conffile = NULL;
	is_stdin = 0;
	return (0);
}

/***
 * Function:
 *	common_closefile	- close a scanned file
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Closes the file pointed to by fp.
 */
int
common_closefile(void)
{

	if (conffile != NULL) {
		if (!is_stdin)
			fclose(conffile);
		conffile = NULL;
	}
	return (0);
}

/***
 * Function:
 *	confgetline		- read a line from a file
 * Inputs:
 *	fp			- the file to read from
 *	line			- a pointer to a location to store the line
 *	n			- the location to store the line length
 * Returns:
 *	The line pointer on success, NULL on error or EOF.
 * Modifies:
 *	Reallocates a buffer at *line to as much as needed.
 */
char *
confgetline(FILE *fp, char **line, size_t *len)
{
#ifdef HAVE_GETLINE
	ssize_t r;
#else
#ifdef HAVE_FGETLN
	char *p, *q;
	size_t n;
#else
#error Either HAVE_GETLINE or HAVE_FGETLN must be defined, check the Makefile!
#endif
#endif

#ifdef HAVE_GETLINE
	r = getline(line, len, fp);
	if (r == -1)
		return (NULL);
	while (r > 0 && ((*line)[r - 1] == '\r' || (*line)[r - 1] == '\n'))
		(*line)[--r] = '\0';
	return (*line);
#else
#ifdef HAVE_FGETLN
	p = fgetln(fp, &n);
	if (p == NULL)
		return (NULL);
	while (n > 0 && (p[n - 1] == '\r' || p[n - 1] == '\n'))
		n--;
	if (*line == NULL || *len < n + 1) {
		q = realloc(*line, n + 1);
		if (q == NULL)
			return (NULL);
		*line = q;
		*len = n + 1;
	} else {
		q = *line;
	}
	memcpy(q, p, n);
	q[n] = '\0';
	return (q);
#endif
#endif
}
