/* 
 * libCW- 2.0 - Copyright (C) 2006 -> 2014 by Ted Williams - WA0EIR
 * Derived mostly from qrq.c - High speed morse trainer
 * by Fabian Kurz, DJ1YFK, Copyright (C) 2006, and
 * a few bits from unixcw
 * by Simon Baldwin, G0FRD.Copyright (C) 2001-2006
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */ 

/*
 * include file for libCW.h
 */
#include "libCW.h"

#include <errno.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/kd.h>
#include <sys/msg.h>
#include <sys/ioctl.h>
#include <linux/rtc.h>

#include <pulse/simple.h>
#include <pulse/error.h>

/*
 * set these as needed by the application
 */
#define TESTMODE  0
#define MESG      0               /* 0=no message, 1=with message support */
#define DEBUG     0

#define FUDGE     1.000           /* no fudge factor */
#if 0
#define FUDGE     1.066           /* fudge factor for my lousy soundcard */
#endif

#define MAX_NAME  20
#define MAX_LETTERS 512
#define MAX_WORD_LENGTH  100L     /* Maximum length of a word */
#define PI 3.1415926              /* Indiana residents, feel free to */
                                  /* change it to 3.2 */
#define DIT '.'
#define DAH '-'
#define DEFAULT_WPM   20.0
#define MIN_WPM        5.0
#define FARNS_OFF  MIN_WPM - 1.0  /* farnsworth off */
#define RISE 10.0
#define FALL 10.0

#define SAMPLERATE 8000
//#define SAMPLERATE 44100

#define TRUE 1
#define FALSE !TRUE
#define SENT_MSG 7L               /* as defined in common.h */

/*
 * CWirc stuff
*/
#if WITH_CWIRC == 1
#define AUDIOBUF_SIZE 22050   i   /* Samples */
struct cwirc_extension_api {
   int semid;
   signed short int out_audiobuf[AUDIOBUF_SIZE];
   int out_audiobuf_start;
   int out_audiobuf_end;
   int in_key;
   int pid;
   };
extern struct cwirc_extension_api *cwirc_api_shm;
#endif

#if MESG == 1
extern int qid;
#endif

/* set some variables to their default values */
char serdevice[MAX_NAME] = "/dev/ttyS0";       /* default serial device path */
int  ser_fd = 0;
char pa_name[MAX_NAME];                        /* shows up on pa vol ctrl */
int keyMode = RTS;                             /* PA, RTS, or CWIRC */
int doReply = FALSE;                           /* send reply msg when true */

int rts_on = TIOCM_RTS;                        /* for key down */
int rts_off = 0;                               /* for key up */
int initialized = 0;                           /* set to 1 by cw_init */

/* for PA */
static pthread_attr_t cwattr;                 /* thread attributes */
static pthread_t cwthread = 0;                /* thread for CW output */
static const pa_sample_spec ss =
   {
      .format = PA_SAMPLE_S16LE,
      .rate = SAMPLERATE,
      .channels = 2
   };

   pa_simple *sp = NULL;

int rtc_rate = 64;                          /* RTC interupt rate */

char *codetable[MAX_LETTERS];

static int freq = 700;                        /* freq value for SC */
static long samplerate = SAMPLERATE;

float wpm = DEFAULT_WPM;                      /* current speed to default */
float farns_wpm = FARNS_OFF;                  /* farnsworth speed to default */

float dit_time = 1200.0 / DEFAULT_WPM;
float dah_time = 1200.0 / DEFAULT_WPM * 3;    /* dah time in milliseconds */
float farns_time;                             /* farns time */

float ic_time = 1200 / DEFAULT_WPM;           /* intra letter time */
float il_time = 3 * 1200 / DEFAULT_WPM;       /* inter letter time */
float iw_time = 7 * 1200 / DEFAULT_WPM;       /* inter word time */

float rise = RISE;                            /* risetime in ms */
float fall = FALL;                            /* fall time in ms */
float rt = SAMPLERATE / 1000 * RISE;          /* normalized risetime */
float ft = SAMPLERATE / 1000 * FALL;          /* normalized falltime */

#if WITH_CWIRC == 1
extern struct cwirc_extension_api *cwirc_api_shm;
struct cwirc_extension_api *cwirc_api_shm;
#endif

/* forward declarations internal functions */
void *send_cw (void *p);
void keySC (char *code);
void keyRTS (char *code);
int  open_serial (void);
int  timer (float usec);
int  close_devs(void);
int  tonegen (int freq, float length);

#if WITH_CWIRC == 1
void keyCWIRC (char *code);
#endif


/*
 * cw_init function
 * if name and ser args are present, copy them to local vars.
 * then, set the mode to PA, RTS or CWIRC. PA needs to create a new pa stream.
 * RTS needs to open the serial card.  CWIRC just works. Then init the
 * character table and setup traps.  All need the thread created.
 */
int cw_init (int newmode, char name[], char ser[])
{
   int i, rtn;

   /*join the thread if already initialized */
   if (initialized == 1)
   {
      pthread_join (cwthread, NULL); 
   }

   /* check for a name argument */
   if (name == NULL)
   {
      fprintf (stderr, "cw_init Warning: name argument is NULL\n");
   }
   else
      strncpy (pa_name, name, MAX_NAME - 1);      /* copy name to pa_name */

   /* check for a serial device */
   if (ser == NULL)
   {
      fprintf (stderr, "cw_init Warning: serial device argument is NULL\n");
   }
   else
   {
      strncpy (serdevice, ser, MAX_NAME - 1) ;    /* copy ser to serdevice */
   }

   /* set newmode to PA or RTS */
   switch (newmode)
   {
      case PA:
         #if DEBUG == 1
         fprintf (stderr, "CW_INIT - TO PA MODE\n");
         #endif
         
         if (cw_set_keyMode (PA, name) != PA)     /* set keyMode to PA */
         {
            return 0;                             /* set keyMode failed */
         }
         break;

      case RTS:

         if ((cw_set_keyMode (RTS, ser)) == RTS)
         {
            strncpy (serdevice, ser, MAX_NAME - 1); /* save the ser dev */ 
         }
         #if DEBUG == 1
         fprintf (stderr, "CW_INIT - TO RTS MODE ON %s\n", serdevice);
         #endif
         break;

      #if WITH_CWIRC == 1
      case CWIRC:
         #if DEBUG == 1
         fprintf (stderr, "USING CWIRC\n");
         #endif
         keyMode = CWIRC;
         break;
      #endif

      default:   /* invalid mode */
         fprintf (stderr, "cw_init error: invalid mode %d\n", newmode);
         exit (1);
   }


   /* Initialize the codetable array */
   /* clear the array so unused chars become spaces */
   if (initialized == 0)
   {
      for (i = 1; i < MAX_LETTERS; i++)
      {
         codetable[i] = " ";
      }

      /* Pro Signs and Special */        /* Shift/Alt  */
      codetable[1]         = "......";   /* F1 = Error */
      codetable[2]         = ".-.-.";    /* F2 = <AR>  */
      codetable[3]         = "-.--.";    /* F3 = <KN>  */
      codetable[4]         = "...-.-";   /* F4 = <SK>  */
      codetable[5]         = ".-...";    /* F5 = WAIT  */
      codetable[6]         = "-...-.-";  /* F6 = BREAK */

      /* Numbers */
      codetable[(int) '1'] = ".----";
      codetable[(int) '2'] = "..---";
      codetable[(int) '3'] = "...--";
      codetable[(int) '4'] = "....-";
      codetable[(int) '5'] = ".....";
      codetable[(int) '6'] = "-....";
      codetable[(int) '7'] = "--...";
      codetable[(int) '8'] = "---..";
      codetable[(int) '9'] = "----.";
      codetable[(int) '0'] = "-----";
      codetable[216]       = "-----";   /* 216 or 330h is a zero with a slash */

      /* Alphabet */
      codetable[(int) ' '] = " ";
      codetable[(int) 'A'] = ".-";
      codetable[(int) 'B'] = "-...";
      codetable[(int) 'C'] = "-.-.";
      codetable[(int) 'D'] = "-..";
      codetable[(int) 'E'] = ".";
      codetable[(int) 'F'] = "..-.";
      codetable[(int) 'G'] = "--.";
      codetable[(int) 'H'] = "....";
      codetable[(int) 'I'] = "..";
      codetable[(int) 'J'] = ".---";
      codetable[(int) 'K'] = "-.-";
      codetable[(int) 'L'] = ".-..";
      codetable[(int) 'M'] = "--";
      codetable[(int) 'N'] = "-.";
      codetable[(int) 'O'] = "---";
      codetable[(int) 'P'] = ".--.";
      codetable[(int) 'Q'] = "--.-";
      codetable[(int) 'R'] = ".-.";
      codetable[(int) 'S'] = "...";
      codetable[(int) 'T'] = "-";
      codetable[(int) 'U'] = "..-";
      codetable[(int) 'V'] = "...-";
      codetable[(int) 'W'] = ".--";
      codetable[(int) 'X'] = "-..-";
      codetable[(int) 'Y'] = "-.--";
      codetable[(int) 'Z'] = "--..";

      /* Punctuation */
      codetable[(int) '='] = "-...-";
      codetable[(int) '/'] = "-..-.";
      codetable[(int) '?'] = "..--..";
      codetable[(int) ','] = "--..--";
      codetable[(int) '.'] = ".-.-.-";
      codetable[(int) ':'] = "---...";
      codetable[(int) '\''] = ".----.";
      codetable[(int) '"'] = ".-..-.";
      codetable[(int) '_'] = "..--.-";
      codetable[(int) '('] = "-.--.";
      codetable[(int) ')'] = "-.--.-";
      codetable[(int) '#'] = "-.---";
      codetable[(int) '-'] = "-...-";
      codetable[(int) '|'] = "...-..";
      codetable[(int) '\\'] = "-.....";
      codetable[(int) '*'] = "-----.";
      codetable[(int) ';'] = "-.-.-.";
      codetable[(int) '@'] = ".--.-.";
      codetable[(int) '^'] = "....--.-.";
      codetable[(int) '$'] = "...-..-";
      codetable[(int) '!'] = "....-.";
      codetable[(int) '>'] = "....---.";
      codetable[(int) ']'] = "....-....";
      codetable[(int) '['] = "....-..";
      codetable[(int) '<'] = "....-.-..";
      codetable[(int) '&'] = "....--.";
      codetable[(int) '%'] = "....-.--.";
      codetable[(int) '~'] = "....--";
      codetable[(int) '+'] = ".-.-.";
      codetable[(int) '{'] = "....-.--";
      codetable[(int) '}'] = "....--..-";

      /* Initialize the morse thread */
      pthread_attr_init (&cwattr);
      pthread_attr_setdetachstate (&cwattr, PTHREAD_CREATE_JOINABLE);

      /*
       * trap the following signals when exiting
       */
      signal (SIGINT,  (void *)cw_exit);
      signal (SIGTERM, (void *)cw_exit);
      signal (SIGQUIT, (void *)cw_exit);
      signal (SIGSEGV, (void *)cw_exit);

      /* create the cwthread and set initialized */
      rtn = pthread_create (&cwthread, NULL, &send_cw, (void *) "");
      if (rtn < 0)
      {
         fprintf (stdout, "cw_init: pthread_create error\n");
         return 0;
      }
      initialized = 1;
      return 1;
   }
}


/*
 * cw_exit
 * Make sure the sidetone is off and the key is up, then exit
 */
void cw_exit (void)
{
   signal (SIGINT,  SIG_IGN);
   signal (SIGTERM, SIG_IGN);
   signal (SIGQUIT, SIG_IGN);
   signal (SIGSEGV, SIG_IGN);
   fprintf(stderr, "libCW terminating\n");

   if (ser_fd != 0)
   {
      if (ioctl (ser_fd, TIOCMSET, &rts_off) == -1)
      {
         perror("cw_exit: RTS key up failed");
      }
   }

   if (pthread_kill (cwthread, 9) == 0)
   {
      perror ("try1");
      fprintf (stderr, "Thread killed\n");
   }

   /* close pulseaudio */
   pa_simple_free (sp);
   exit (73);
}


/*
 * close_devs - closes ttyS after making sure the key is up.
 */
int close_devs (void)
{

   if (ser_fd != 0)                      /* if ser_fd is opened */
   {
      /* make sure the key is up */
      if (ioctl (ser_fd, TIOCMSET, &rts_off) < 0)
      {
         perror ("close_devs: RTS key up failed");
         return (0);
      }
      close (ser_fd);                    /* close ser device */
      ser_fd = 0;
   }
   return (1);
}


/*
 * cw_send_str
 * joins the cwthread to send a string of characters
 * called by all other cw_send_* functions.
 */
int cw_send_str (char *str)
{
   int rtn;
   static char s[MAX_WORD_LENGTH];

   /* cw_init must be called first */
   if (initialized == 0)
   {
      fprintf (stderr, "cw_send_str: cw_init must be called first.\n");
      return 0;
   }

   /* wait for any current thread */
   pthread_join (cwthread, NULL);

   strcpy (s, str);                 /* save the string */
   rtn = pthread_create (&cwthread, NULL, &send_cw, (void *) s);
   if (rtn < 0)
   {
      fprintf (stdout, "cw_send_str: pthread_create error\n");
      return 0;
   }
   return 1;
}


/*
 * cw_send_word
 * this does the same as send str, but adds a space to the end
 * to make it compatable for twcw.
 */
int cw_send_word (char *str)
{
   static char new_str[MAX_WORD_LENGTH +1 ];

   strcpy (new_str, str);
   strcat (new_str, " ");
   cw_send_str (new_str);
   return (1);
}


/*
 * cw_send_char - sends one char
 * This method causes a little extra overhead and the timing
 * is off slightly - abt 1 sec longer than it should be over 60 secs. 
 */
int cw_send_char (char ch)
{
   char str[2];

   str[0] = ch;
   str[1] = '\0';

   cw_send_str (str);
   return (1);
}


/*
 * send_cw
 * started as thread by cw_send_str() or cw_send_word()
 */
void *send_cw (void *p)
{
   int i=0;
   char *pt;
   char *code;

   #if MESG == 1
   struct tag1a
   {
      long type;
      union
      {
         char word[MAX_WORD_LENGTH];
         int  val;
      } data;
   } reply;
   #endif

   if (strlen ((char *)p) == 0)     /* nothing to send */
   {
      pthread_exit (0);
   }

   /* open the serial device if needed */
   if (keyMode == RTS)
   {
      ser_fd = open_serial ();
      if (ser_fd < 0)
      {
         fprintf (stderr, "send_cw: open %s failed\n", serdevice);
         pthread_exit (0);
      }
   }

   pt = (char *) p;
   /* now run thru the string, and call keySC, keyTimer or keyCWIRC */
   while (pt[i] != '\0')
   {
      /* get the ./- representation for the character */
      code = codetable[(int)toupper((int)pt[i])];

      /* key devices */
      if ((keyMode & PA) == PA)
      {
         keySC (code);
      }
      if ((keyMode & RTS) == RTS)
      {
         keyRTS(code);
      }

#if WITH_CWIRC == 1
      if (keyMode == CWIRC)
      {
         keyCWIRC(code);
      }
#endif

#if MESG == 1
      /* if doReply is true and MESG is 1, we need to send a reply message */
      if (doReply == TRUE)
      {
         reply.type = SENT_MSG;
         reply.data.word[0] = pt[i];
         reply.data.word[1] = '\0';
         if (msgsnd (qid, (struct msgbuf *)&reply,
                     sizeof(reply.data), 0) == -1)
         {
            perror ("send_cw: msgsnd SENT_MSG failed");
            exit (1);
         }
      }
#endif
      i++;
   }
   close_devs ();
   pthread_exit (0);
}


/*
 * keySC (char *code)
 * uses the morse representation to key the SC
 */
void keySC (char *code)
{ 
   int i = 0;
   float t;

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                /* do a space or non-valid char */
      {                                  /* this assumes that the */
         tonegen (0, iw_time - il_time); /* previous character has */
         return;                         /* done the il_time already */
      }

      if (code[i] == DIT)
      {
         t = dit_time;
      }
      else if (code[i] == DAH)
      {
         t = dah_time;
      }
      else
      {
         fprintf (stderr, "keySC: Program error - not a DIT or DAH\n");
         exit (1);
      }

      tonegen (freq, t);                 /* and send the dit/dah */

      /* do ic_time if there are more elements in this letter */
      i++;
      if (code[i] != '\0')               /* if more ./- to do */
      {                                  /* for this character */ 
         tonegen (0, ic_time);           /* add intra character time */
      }
   }
   tonegen (0, il_time);                 /* do inter letter time */
   return;
}


/*
 * keyRTS (char code)
 * uses the morse representation to key the RTS
 */
void keyRTS (char *code)
{ 
   int i = 0;
   float t;

   if (ser_fd == 0)  /* if no serial device */
   {
      return;
   }

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                      /* do a space or non-valid ch */
      {
         timer (iw_time - il_time);
         return;
      }

      if (code[i] == DIT)
      {
         t = dit_time;
      }
      else
      {
         t = dah_time;
      }

      if (ioctl (ser_fd, TIOCMSET, &rts_on) < 0)    /* key down */
      {
         perror("keyRTS: RTS key down failed");
      }

      timer (t);

      if (ioctl (ser_fd, TIOCMSET, &rts_off) < 0)   /* RTS key up */
      {
         perror ("keyRTS: RTS key up failed");
      }
      i++;
      if (strlen (code) > i)                   /* if more ./- to do */
      {                                        /* for this character */ 
         timer (ic_time);
      }
   }
   timer (il_time);
}


/*
 * keyCWIRC
 * toggles cwirc_api_shm->in_key to key cwirc
 */
#if WITH_CWIRC == 1
void keyCWIRC (char *code)
{
   int i = 0;
   float t;

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                /* do a space or non-valid ch */
      {
         timer (iw_time - il_time);
         return;
      }

      if (code[i] == DIT)
         t = dit_time;
      else
         t = dah_time;

      /* key cwirc */
      cwirc_api_shm->in_key = 1;
      timer (t);
      cwirc_api_shm->in_key = 0;

      i++;
      if (strlen (code) > i)             /* if more ./- to do */
      {                                  /* for this character */ 
         timer (ic_time);
      }
   }
   timer (il_time);
   return;
}
#endif


/*
 * tonegen
 * This function was mostly taken from qrq.
 * Copyright (C) 2006  Fabian Kurz, DJ1YFK.  Thanks, Fabian.
 * Generates a sinus tone of frequency 'freq' and length 'len'
 * based on 'samplerate', 'rt' (rise time), 'ft' (fall time)
 */
int tonegen (int freq, float len)
{
   int x, rtn;
   int buf[20000];   //plenty big for a 8wpm dah
   float val = 0.0;
   int error;

   /* convert len from milliseconds to samples and determine rise/fall time */
   /* include FUDGE factor */
   len = samplerate * (len * FUDGE) / 1000.0;

   for (x=0; x < len-1; x++)
   {
      /* if freq == 0, then quietly fill the buffer with zeros */
      if (freq == 0)                             /* quiet time */
      {
         buf[x] = 0;
         buf[x] = buf[x] + (buf[x]<<16);
      }
      else
      {
         val = sin(2*PI*freq*x/samplerate);
         if (x < rt)                             /* within rise time */
         {
            val = val * sin(PI*x/(2.0*rt));
         }

         if (x > (len-ft))                       /* within fall time */
         {
            val = val * sin(2*PI*(x-(len-ft)+ft)/(4.0*ft));
         }
         buf[x] = val * 32000;
         buf[x] = buf[x] + (buf[x]<<16);         /* second channel */
      }
   }

   /* now write the buffer */
   rtn = pa_simple_write (sp, buf, 4*x, &error); 
   if (rtn == -1)
   {
      fprintf (stderr, "pa_simple_write failed - error=%d: %s\n",
               error, pa_strerror(error));
      return -1;
   }
   return 0;
}


/*
 * timer - use the real time clock for accurate timing
 */
int timer (float msec)
{
   unsigned long rtc_data;
   int fd;
   int i;
   int cnt;

   cnt = msec * rtc_rate / 1000.0;
   /* open rtc */
   if ((fd = open ("/dev/rtc", O_RDONLY)) < 0)
   {
      perror ("timer: open /dev/rtc failed");
      return -1;
   }

   /* configure rtc for periodic interupts */
   if (ioctl (fd, RTC_IRQP_SET, rtc_rate) < 0)
   {
      perror ("timer: RTC_IRQP_SET");
      return -1;
   }

   if (ioctl (fd, RTC_PIE_ON, 0) < 0)
   {
      perror ("timer: PIE_ON");
      return -1;
   }

   i = 0;
   while (cnt - i > 0)
   {
      if (read (fd, &rtc_data, sizeof(rtc_data)) < 0)
      {
         perror ("timer: RTC read");
         exit (1);
      }
      i = i + (rtc_data >> 8);
   }

   if (ioctl (fd, RTC_PIE_OFF, 0) < 0)
   {
      perror ("timer: PIE_OFF");
      exit (1);
   }
   close (fd);
   return 0;
}


/*
 * cw_set_doReply - sets/clears reply mode 
 * set doReply to TRUE/FALSE
//TJW join thread?
 */
int  cw_set_doReply (int new_doReply)
{
   doReply = new_doReply;
   return (doReply);
}


/*
 * get doReply - returns current reply mode
//TJW join thread
 */
int  cw_get_doReply (void)
{
   return (doReply);
}


/*
 *cw_get_serdev - returns the current key mode
//TJW join thread?
 */
char *cw_get_serdev (void)
{
   if (serdevice != NULL)
   {
      return (serdevice);
   }
   else
   {
      return (NULL);
   }
}


/*
 * cw_get_keyMode
//TJW join thread
 */
int cw_get_keyMode (void)
{
   return (keyMode);
}


/*
 * cw_set_keyMode (int newmode, char *name)
 * Use PA, RTS, or IRC for newmode.  Returns keyMode, if it works
 */
int cw_set_keyMode (int newmode, char *name)
{  
   int fd, error;

#if WITH_CWIRC == 1
   /* if current mode is CWIRC, you're stuck there. just return */
   if (newmode == CWIRC)
   {
      return keyMode;
   }
#endif

   /* if initialized, wait for the current thread */
   if (initialized == 1)
   {
      pthread_join (cwthread, NULL);
   }

   /* setup the mode */
   switch (newmode)
   {
      case PA:
         /* check for a soundcard pointer */
         if (sp == NULL)          /* don't create a sp if we have one */
         {
            sp = pa_simple_new (NULL, pa_name, PA_STREAM_PLAYBACK, NULL,
                                name, &ss, NULL, NULL, &error);
            if (sp == NULL)       /* if NULL now, we have an error */
            {
               perror ("cw_set_keyMode: pa_simple_new failed");
               return 0;
            }
         }

         #if DEBUG == 1
         fprintf (stderr,
                  "set_keyMode PA wpm=%2.0f/%2.0f dit=%4.1f dah=%4.1f "
                  "il=%4.1f iw=%4.1f rt/ft=%4.1f\n",
                  wpm, farns_wpm, dit_time, dah_time, il_time, iw_time, rt);
         #endif
         break;

      case RTS:
         /* if serdev is NULL, we can't do RTS, so fail */
         if (strlen (name) == 0)
         {
            printf ("Gotta fail\n");
         }

         /* open rtc and find the highest interupt rate */
         rtc_rate = 4096;
         if ((fd = open ("/dev/rtc", O_RDONLY)) < 0)
         {
            perror ("cw_set_keyMode: open /dev/rtc failed");
            exit (1);
         }
         /* find the highest interupt rate */
         /* if 64 is the system default rate, that's kinda slow */
         while (ioctl(fd, RTC_IRQP_SET, rtc_rate) < 0)
         {
            fprintf (stderr, "cw_set_keyMode RTC_IRQP_SET %d failed\n",
                     rtc_rate);
            rtc_rate = rtc_rate/2;
         }
         fprintf (stderr, "RTC rate = %d\n", rtc_rate);
         close (fd);   /* close the rtc */

         #if DEBUG == 1
         fprintf (stderr,
                  "set_keyMode RTS wpm=%2.0f/%2.0f dit=%4.1f dah=%4.1f "
                  "il=%4.1f iw=%4.1f rt/ft=%4.1f\n",
                  wpm, farns_wpm, dit_time, dah_time, il_time, iw_time, rt);
         #endif

         break;

      default:
         fprintf (stderr, "cw_set_keyMode: invalid mode %d\n", newmode);
         return 0;       /* oops!  invalid mode */
   }
   keyMode = newmode;
   return keyMode;
}


/* 
 * cw_get_wpm - returns current wpm
//TJW join thread?
 */
int cw_get_wpm (void)
{
   return (wpm);
}


/*
 * cw_set_wpm
 * set wpm.  Calculate the times for all character, including the
 * iw_time and ic_time for farnsworths, if required. Also figure the
 * rise/fall times
 */
int cw_set_wpm (int new_wpm)
{
   float wpm_time;

   /* cw_init must be called first */
   if (initialized == 0)
   {
      fprintf (stderr, "cw_set_wpm: cw_init must be called first.\n");
      return -1;
   }
   /*
    * we don't want to change speed while sending somethingm
    * so wait for the thread
    */
   pthread_join (cwthread, NULL);

   /*
    * save the new wpm value if it is >= 5
    */
   if (new_wpm >= MIN_WPM)    //TJW and =< MAX_WPM
   {
      wpm = new_wpm;
   }
   /*
    * speed is in wpm so we have to calculate the dot-length in
    * milliseconds using the well-known formula -> dotlength = 1200 / wpm
    */
   dit_time = 1200.0 / wpm;
   dah_time = 3.0 * dit_time;
   ic_time = dit_time;

   if ((farns_wpm < wpm) && (farns_wpm > FARNS_OFF))
   { /* do farnsworth timing */
      farns_time = 1200 / farns_wpm * 50.0; /* time for a "word" @ fwpm */ 
      wpm_time = 31 * dit_time;             /* PARIS minus iw_time @ wpm */
      il_time = 3.0/19.0 * (farns_time - wpm_time);
      iw_time = 7.0/19.0 * (farns_time - wpm_time);
   }
   else
   {  /* otherwise use standard timing */
      il_time = 3 * dit_time;
      iw_time = 7 * dit_time;
   }

   /* rt and ft are the rise and fall times in milliseconds */
   rt = (samplerate / 1000.0) * rise;
   ft = (samplerate / 1000.0) * fall;

#if DEBUG == 1
   fprintf (stderr,
            "cw_set_wpm  wpm=%2.0f/%2.0f dit=%4.1f dah=%4.1f "
            "il=%4.1f iw=%4.1f rt/ft=%4.1f\n",
            wpm, farns_wpm, dit_time, dah_time,
            il_time, iw_time, rt);
   #endif
   return wpm;
}


/*
 * cw_set_freq - sets the frequency of the audio out
//TJW join thread
 */
int cw_set_freq (int new_freq)
{
   freq = new_freq;
   return freq;
}
   

/*
 * cw_get_freq - gets the frequency of the audio out
//TJW join thread
 */
int cw_get_freq (void)
{
   return freq;
}


/* 
 * cw_get_fwpm - returns current farnsworth wpm
//TJW join thread
 */
int cw_get_fwpm (void)
{
   return (farns_wpm);
}


/*
 * cw_set_fwpm
 * set farns_wpm.  Calculate the farnsworth times for il_time and
 * iw_time for farnsworth.
 */
int cw_set_fwpm (int new_fwpm)
{
   float wpm_time;

   if (initialized == 0)
   {
      fprintf (stderr, "cw_set_fwpm: cw_init must be called first.\n");
      return -1;
   }

   /*
    * we don't want to change speed while sending something
    * so wait for the thread if it is running.
    */
   pthread_join (cwthread, NULL);

   /*
    * speed is in wpm so we have to calculate the dot-length in
    * milliseconds using the well-known formula -> dotlength = 1200 / wpm
    */
   dit_time = 1200.0 / wpm;
   dah_time = 3.0 * dit_time;
   ic_time = dit_time;

   if ((new_fwpm < wpm) && (new_fwpm > FARNS_OFF))
   { /* do farnsworth timing */
      farns_wpm = new_fwpm;
      farns_time = 1200 / farns_wpm * 50.0; /* time to send a "word" @ fwpm */ 
      wpm_time = 31 * dit_time;             /* PARIS minus iw_time @ wpm */
      
      il_time = 3.0/19.0 * (farns_time - wpm_time);
      iw_time = 7.0/19.0 * (farns_time - wpm_time);
   }
   else
   {
      il_time = 3 * dit_time;
      iw_time = 7 * dit_time;
   }
   rt = (samplerate / 1000.0) * rise;
   ft = (samplerate / 1000.0) * fall;

   #if DEBUG == 1
   fprintf (stderr, "cw_set_fwpm  wpm=%2.0f/%2.0f dit=%4.1f dah=%4.1f "
            "il=%4.1f iw=%4.1f\n",
            wpm, farns_wpm, dit_time, dit_time*3, il_time, iw_time);
   fprintf (stderr, "SAMPLERATE = %ld\n", samplerate);
   #endif
   return farns_wpm;
}


/*
 * open_serial
 * open the serial device and return fd. 
 */
int open_serial ()
{
   int fd = 0;

   if (ser_fd != 0)         /* already open ?*/
   {
      /* yes, so just make sure the key is up and return ser_fd */
      ioctl (fd, TIOCMSET, &rts_off);
      return (ser_fd);
   }

   if ((fd = open(serdevice, O_RDWR, 0)) < 0)
   {
      perror("open_serial: can't open the serial device");
   }
          else
   {
      /* make sure the key is up */
      if (fd > 0)
      {
         ioctl (fd, TIOCMSET, &rts_off);
      }
   }
   return fd;   /* return fd or error */
}


#if TESTMODE == 1
/*
 * main function for testing standalone
 */
int main (int argc, char *argv[])
{
   int i, j;
   char d;
   char pgmName[] = "test mode";
   char newdev[] = "/dev/ttyS0";

   /*
    * select one key mode
    */
   //int mode = RTS;
   int mode = PA;
   //int mode = CWIRC;

   /* initialize libCW */
   cw_init (mode, pgmName, newdev);

   /* set some other setup functions */
   cw_set_doReply (FALSE);                 /* turn reply off */
   cw_set_wpm (20);                        /* set wpm */
   //cw_set_fwpm (FARNS_OFF);              /* set fwpm */
   //cw_set_fwpm (10);                     /* set fwpm */

   printf ("Enter to start: ");          /* hit rtn when ready */
   d = getchar ();

   /*
    * feel free to add what ever other ways you want to 
    * test libCW here
    */

#if 0
   /*
    * send paris 20 times as chars (twcw style)
    */
   for (i=0; i<20; i++)
   {
      cw_send_char ('p');
      cw_send_char ('a');
      cw_send_char ('r');
      cw_send_char ('i');
      cw_send_char ('s');
      cw_send_char (' ');
   }
#endif

#if 1
   /*
    * send paris 5 times as words (twcw style)
    */
   for (i=0; i<5; i++)
   {
      cw_send_word ("paris");
   }
#endif

   pthread_join (cwthread, NULL);       /* wait for it before we terminate */
//cw_exit ();
   return 0;
}
#endif   /* END TESTMODE */

