///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/ViewportManager.h>
#include "MultiFileWriter.h"

namespace AtomViz {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(MultiFileWriter, AbstractFileColumnWriter)
DEFINE_PROPERTY_FIELD(MultiFileWriter, "UseWildcardFilename", _useWildcardFilename)
DEFINE_PROPERTY_FIELD(MultiFileWriter, "WildcardFilename", _wildcardFilename)
DEFINE_PROPERTY_FIELD(MultiFileWriter, "StartFrame", _startFrame)
DEFINE_PROPERTY_FIELD(MultiFileWriter, "EndFrame", _endFrame)

/******************************************************************************
 * Exports the atoms in the given scene to the output file.
 *****************************************************************************/
bool MultiFileWriter::exportAtoms(DataSet* dataset, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(dataset);
	OVITO_ASSERT_MSG(!outputFile().isEmpty(), "MultiFileWriter::exportAtoms()", "Output filename has not been set. AtomsFileWriter::setOutputFile() must be called first.");
	OVITO_ASSERT_MSG(startFrame() <= endFrame(), "MultiFileWriter::exportAtoms()", "Export interval has not been set. MultiFileWriter::setExportInterval() must be called first.");
	
	if(startFrame() > endFrame())
		throw Exception(tr("The animation interval to be exported is empty or has not been set."));
		
	ViewportSuspender noVPUpdates;
	
	if(useWildcardFilename()) {
		if(wildcardFilename().isEmpty())
			throw Exception(tr("Cannot write animation to multiple atoms files. No wild-card pattern has been set."));
		if(wildcardFilename().contains(QChar('*')) == false)
			throw Exception(tr("Cannot write animation to multiple atoms files. The wild-card pattern must contain at least one * placeholder."));
		
		QDir dir = QFileInfo(outputFile()).dir();
		
		// Export each animaion frame to a new file.
		for(int frame = startFrame(); frame <= endFrame(); frame++) {
			
			// Generate an output filename from the wild-card pattern.
			QString filename = dir.absoluteFilePath(wildcardFilename());
			filename.replace(QChar('*'), QString::number(frame));
			
			TimeTicks time = frameToTime(frame, dataset->animationSettings()->ticksPerFrame());
			QVector<TimeTicks> singleTime(1, time);
			if(!writeAtomsFile(filename, dataset, singleTime, suppressDialogs))
				return false;
		}
	}
	else {
		
		// The list of animation frames to be exported.
		QVector<TimeTicks> exportFrames;
		
		// Iterate over all animation frames of the export interval.
		for(int f = startFrame(); f <= endFrame(); f++) {
			TimeTicks time = frameToTime(f, dataset->animationSettings()->ticksPerFrame());
			exportFrames.push_back(time);
		}
		
		// Write one big file containing all animation frames.
		if(!writeAtomsFile(outputFile(), dataset, exportFrames, suppressDialogs))
			return false;
	}
	
	return true;
}

};	// End of namespace AtomViz
