///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/ObjectLoadStream.h>
#include <core/reference/PropertyFieldDescriptor.h>
#include <core/plugins/Plugin.h>

namespace Core {

/******************************************************************************
* Opens the stream for reading.
******************************************************************************/
ObjectLoadStream::ObjectLoadStream(QDataStream& source) : LoadStream(source), currentObject(NULL)
{
	qint64 oldPos = filePosition();

	// Go to index at the end of the file.
	setFilePosition(source.device()->size() - 2 * (qint64)(sizeof(qint64) + sizeof(quint32)));
	// Read index.
	qint64 beginOfRTTI, beginOfObjTable;
	quint32 classCount, objCount;
	*this >> beginOfRTTI;
	*this >> classCount;
	classes.resize(classCount);
	*this >> beginOfObjTable;
	*this >> objCount;
	objects.resize(objCount);

	// Go to start of class table.
	setFilePosition(beginOfRTTI);
	expectChunk(0x200);
	for(int i=0; i<classes.size(); i++) {
		// Load the runtime type information from the stream.
		ClassEntry& classEntry = classes[i];
		expectChunk(0x201);
		classEntry.descriptor = PluginClassDescriptor::loadRTTI(*this);
		if(classEntry.descriptor->isSerializable() == false)
			throw Exception(tr("Failed to load non-serializable class %1.").arg(classEntry.descriptor->name()));
		closeChunk();

		// Load the plugin containing the class now.
		classEntry.descriptor->plugin()->loadPlugin();

		// Read the stored property fields for this class from the stream.
		expectChunk(0x202);
		for(;;) {
			quint32 chunkId = openChunk();
			if(chunkId == 0x00000000) {
				closeChunk();
				break;	// end of list
			}
			if(chunkId != 0x00000001)
				throw Exception(tr("File format cannot be read."));

			PropertyFieldEntry propFieldEntry;
			*this >> propFieldEntry.identifier;
			propFieldEntry.definingClass = PluginClassDescriptor::loadRTTI(*this);
			if(classEntry.descriptor->isKindOf(propFieldEntry.definingClass) == false)
				throw Exception(tr("The class hierarchy in the scene file is different from the current class hierarchy."));
			this->readEnum(propFieldEntry.flags);
			*this >> propFieldEntry.isReferenceField;
			if(propFieldEntry.isReferenceField)
				propFieldEntry.targetClass = PluginClassDescriptor::loadRTTI(*this);
			else
				propFieldEntry.targetClass = NULL;
			closeChunk();

			propFieldEntry.field = propFieldEntry.definingClass->findPropertyField(propFieldEntry.identifier.constData());

			// Do a strict check:
			//if(propFieldEntry.field == NULL)
			//	throw Exception(tr("File format error: Did not find property field '%1' in class %2.").arg(propFieldEntry.identifier, propFieldEntry.definingClass->name()));
			if(propFieldEntry.field) {
				if(propFieldEntry.field->isReferenceField() != propFieldEntry.isReferenceField ||
						propFieldEntry.field->isVector() != ((propFieldEntry.flags & PROPERTY_FIELD_VECTOR) != 0) ||
						(propFieldEntry.isReferenceField && propFieldEntry.targetClass->isKindOf(propFieldEntry.field->targetClass()) == false))
					throw Exception(tr("File format error: The type of the property field '%1' in class %2 has changed.").arg(propFieldEntry.identifier, propFieldEntry.definingClass->name()));
			}

			classEntry.propertyFields.push_back(propFieldEntry);
		}
		closeChunk();
	}
	closeChunk();

	// Go to start of object table.
	setFilePosition(beginOfObjTable);
	expectChunk(0x300);
	for(QVector<ObjectEntry>::iterator entry = objects.begin(); entry != objects.end(); ++entry) {
		entry->object = NULL;
		quint32 id;
		*this >> id;
		entry->pluginClass = &classes[id];
		*this >> entry->fileOffset;
	}
	closeChunk();

	// Go back to last position.
	setFilePosition(oldPos);
}

/******************************************************************************
* Loads an object with runtime type information from the stream.
* The method returns a pointer to the object but this object will be
* in an unintialized state until it is loaded at a later time.
******************************************************************************/
PluginClass::SmartPtr ObjectLoadStream::loadObject()
{
	quint32 id;
	*this >> id;
	if(id == 0) return NULL;
	else {
		ObjectEntry& entry = objects[id-1];
		if(entry.object != NULL) return entry.object;
		else {
			// Create an instance of the object class.
			entry.object = entry.pluginClass->descriptor->createInstance(true);
			objectsToLoad.push_back(id-1);
			return entry.object;
		}
	}
}

/******************************************************************************
* Closes the stream.
******************************************************************************/
void ObjectLoadStream::close()
{
	// This prevents re-entrance in case of an exception.
	if(!currentObject) {

		for(int i=0; i<objectsToLoad.size(); i++) {
			quint32 index = objectsToLoad[i];
			currentObject = &objects[index];
			CHECK_OBJECT_POINTER(currentObject->object);

			// Seek to object data.
			setFilePosition(currentObject->fileOffset);

			// Load class contents.
			try {
				try {
					//VerboseLogger() << "Loading object data of " << currentObject->pluginClass->descriptor->name() << endl;
					currentObject->object->setPluginClassFlag(PluginClass::FLAG_OBJ_BEING_LOADED);
					currentObject->object->loadFromStream(*this);
					currentObject->object->clearPluginClassFlag(PluginClass::FLAG_OBJ_BEING_LOADED);
				}
				catch(...) {
					// Clean up.
					CHECK_OBJECT_POINTER(currentObject->object);
					currentObject->object->clearPluginClassFlag(PluginClass::FLAG_OBJ_BEING_LOADED);
					throw;
				}
			}
			catch(Exception& ex) {
				throw ex.appendDetailMessage(tr("Object of class type %1 failed to load.").arg(currentObject->object->pluginClassDescriptor()->name()));
			}
		}

		// Now that all references are in place call post-processing function on each loaded object.
		for(QVector<ObjectEntry>::const_iterator iter = objects.begin(); iter != objects.end(); ++iter)
			iter->object->loadFromStreamComplete();
	}
	LoadStream::close();
}

};
