///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/FloatPropertyUI.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(FloatPropertyUI, NumericalPropertyUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
FloatPropertyUI::FloatPropertyUI(PropertiesEditor* parentEditor, const char* propertyName, const QString& labelText, ParameterUnit* parameterUnit) :  
	NumericalPropertyUI(parentEditor, propertyName, parameterUnit ? parameterUnit : UNITS_MANAGER.floatIdentity(), labelText)
{
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
FloatPropertyUI::FloatPropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField) :  
	NumericalPropertyUI(parentEditor, propField, UNITS_MANAGER.floatIdentity())
{
}

/******************************************************************************
* Takes the value entered by the user and stores it in the property field 
* this property UI is bound to.
******************************************************************************/
void FloatPropertyUI::updatePropertyValue()
{
	if(editObject() && spinner()) {
		if(propertyName()) {
			if(!editObject()->setProperty(propertyName(), spinner()->floatValue())) {
				OVITO_ASSERT_MSG(false, "FloatPropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			editObject()->setPropertyFieldValue(*propertyField(), spinner()->floatValue());						
		}
	}
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void FloatPropertyUI::updateUI()
{
	if(editObject() && spinner() && !spinner()->isDragging()) {
		QVariant val(0.0);
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid() && val.canConvert(QVariant::Double), "FloatPropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to float type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid() || !val.canConvert(QVariant::Double)) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to float type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid());
		}
		spinner()->setFloatValue(val.value<FloatType>());
	}
}

/******************************************************************************
* Gets the minimum value to be entered.
* This value is in native controller units.
******************************************************************************/
FloatType FloatPropertyUI::minValue() const
{
	return (spinner() ? spinner()->minValue() : numeric_limits<FloatType>::min());
}

/******************************************************************************
* Sets the minimum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void FloatPropertyUI::setMinValue(FloatType minValue)
{
	if(spinner()) spinner()->setMinValue(minValue);
}

/******************************************************************************
* Gets the maximum value to be entered.
* This value is in native controller units.
******************************************************************************/
FloatType FloatPropertyUI::maxValue() const
{
	return (spinner() ? spinner()->maxValue() : numeric_limits<FloatType>::max());
}

/******************************************************************************
* Sets the maximum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void FloatPropertyUI::setMaxValue(FloatType maxValue)
{
	if(spinner()) spinner()->setMaxValue(maxValue);
}

};

