///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/IntegerControllerUI.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(IntegerControllerUI, NumericalControllerUI)

/******************************************************************************
* The constructor.
******************************************************************************/
IntegerControllerUI::IntegerControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField) : 
	NumericalControllerUI(parentEditor, refField, UNITS_MANAGER.integerIdentity())
{
}

/******************************************************************************
* Takes the value entered by the user and stores it in the parameter object
* this parameter UI is bound to.
******************************************************************************/
void IntegerControllerUI::updateParameterValue()
{
	IntegerController* ctrl = dynamic_object_cast<IntegerController>(parameterObject());
	if(ctrl != NULL && spinner())
		ctrl->setCurrentValue(spinner()->intValue());
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void IntegerControllerUI::updateUI()
{
	IntegerController* ctrl = dynamic_object_cast<IntegerController>(parameterObject());
	if(ctrl != NULL && spinner() && !spinner()->isDragging()) {
		TimeInterval interval;
		spinner()->setIntValue(ctrl->getCurrentValue());
	}
}

/******************************************************************************
* Gets the minimum value to be entered.
* This value is in native controller units.
******************************************************************************/
int IntegerControllerUI::minValue() const
{
	return (spinner() ? (int)spinner()->minValue() : numeric_limits<int>::min());
}

/******************************************************************************
* Sets the minimum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void IntegerControllerUI::setMinValue(int minValue)
{
	if(spinner()) spinner()->setMinValue(minValue);
}

/******************************************************************************
* Gets the maximum value to be entered.
* This value is in native controller units.
******************************************************************************/
int IntegerControllerUI::maxValue() const
{
	return (spinner() ? (int)spinner()->maxValue() : numeric_limits<int>::max());
}

/******************************************************************************
* Sets the maximum value to be entered.
* This value must be specified in native controller units.
******************************************************************************/
void IntegerControllerUI::setMaxValue(int maxValue)
{
	if(spinner()) spinner()->setMaxValue(maxValue);
}

};

