///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/FloatPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/utilities/ProgressIndicator.h>

#include <atomviz/modifier/analysis/cna/CNAModifier.h>
#include <atomviz/atoms/datachannels/OrientationDataChannel.h>
#include <atomviz/utils/NearestNeighborList.h>
#include <atomviz/utils/OnTheFlyNeighborList.h>

#include "CalculateIntrinsicStrainModifier.h"

namespace CrystalAnalysis {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CalculateIntrinsicStrainModifier, AtomsObjectAnalyzerBase)
DEFINE_REFERENCE_FIELD(CalculateIntrinsicStrainModifier, DeformationGradientDataChannel, "DeformationGradientChannel", deformationGradientChannel)
DEFINE_REFERENCE_FIELD(CalculateIntrinsicStrainModifier, DataChannel, "StrainTensorChannel", strainTensorChannel)
DEFINE_REFERENCE_FIELD(CalculateIntrinsicStrainModifier, DataChannel, "HydrostaticStrainChannel", hydrostaticStrainChannel)
DEFINE_REFERENCE_FIELD(CalculateIntrinsicStrainModifier, DataChannel, "ShearStrainChannel", shearStrainChannel)
DEFINE_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, "LatticeConstant", _latticeConstant)
SET_PROPERTY_FIELD_LABEL(CalculateIntrinsicStrainModifier, _latticeConstant, "Lattice constant")
SET_PROPERTY_FIELD_UNITS(CalculateIntrinsicStrainModifier, _latticeConstant, WorldParameterUnit)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
CalculateIntrinsicStrainModifier::CalculateIntrinsicStrainModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading), _latticeConstant(0)
{
	INIT_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, deformationGradientChannel);
	INIT_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, strainTensorChannel);
	INIT_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, hydrostaticStrainChannel);
	INIT_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, shearStrainChannel);
	INIT_PROPERTY_FIELD(CalculateIntrinsicStrainModifier, _latticeConstant);

	if(!isLoading) {
		deformationGradientChannel = new DeformationGradientDataChannel(DataChannel::DeformationGradientChannel);
		strainTensorChannel = new DataChannel(DataChannel::StrainTensorChannel);
		hydrostaticStrainChannel = new DataChannel(qMetaTypeId<FloatType>(), sizeof(FloatType), 1);
		hydrostaticStrainChannel->setName(tr("Hydrostatic Strain"));
		shearStrainChannel = new DataChannel(qMetaTypeId<FloatType>(), sizeof(FloatType), 1);
		shearStrainChannel->setName(tr("Shear Strain"));
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus CalculateIntrinsicStrainModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	if(deformationGradients() == NULL || strainTensors() == NULL)
		throw Exception(tr("No deformation analysis results available."));

	// Check if it is still valid.
	if(input()->atomsCount() != deformationGradients()->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	CloneHelper cloneHelper;

	// Create a copy of the internal buffer channels and assign them to the output AtomsObject.

	DataChannel::SmartPtr gradClone = cloneHelper.cloneObject(deformationGradients(), true);
	output()->replaceDataChannel(outputStandardChannel(DataChannel::DeformationGradientChannel), gradClone.get());

	DataChannel::SmartPtr strainClone = cloneHelper.cloneObject(strainTensors(), true);
	output()->replaceDataChannel(outputStandardChannel(DataChannel::StrainTensorChannel), strainClone.get());

	DataChannel::SmartPtr hydrostaticStrainClone = cloneHelper.cloneObject(hydrostaticStrains(), true);
	output()->insertDataChannel(hydrostaticStrainClone);

	DataChannel::SmartPtr shearStrainClone = cloneHelper.cloneObject(shearStrains(), true);
	output()->insertDataChannel(shearStrainClone);

	return EvaluationStatus();
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus CalculateIntrinsicStrainModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	// Make sure that the CNA has been performed for the input objectbecause this
	// deformation analysis works only on crystalline lattice atoms.
	expectStandardChannel(DataChannel::CNATypeChannel);

	if(latticeConstant() <= 0.0)
		throw Exception(tr("The lattice constant must be positive."));

	if(calculate(input(), latticeConstant(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Calculates the local deformation of each atom.
******************************************************************************/
bool CalculateIntrinsicStrainModifier::calculate(AtomsObject* atomsObject, FloatType latticeConstant, bool suppressDialogs)
{
	OVITO_ASSERT(latticeConstant > 0.0);
	if(latticeConstant <= 0.0) throw Exception(tr("The reference lattice constant must be positive."));

	// Get the position channel.
	DataChannel* positionChannel = atomsObject->getStandardDataChannel(DataChannel::PositionChannel);
	if(!positionChannel) throw Exception(tr("Input atoms object does not contain a position channel."));

	// Make sure that the displacement analysis has been performed on these atoms.
	DataChannel* cnaChannel = atomsObject->getStandardDataChannel(DataChannel::CNATypeChannel);
	if(!cnaChannel) throw Exception(tr("Input atoms object does not contain a CNA atom type channel. The Common Neighbor Analysis modifier has to be applied first."));

	ProgressIndicator progress(tr("Calculating intrinsic strain tensors."), atomsObject->atomsCount(), suppressDialogs);

	// Prepare the neighbor list.
	OnTheFlyNeighborList neighborList(nearestNeighborList()->nearestNeighborCutoff());
	if(!neighborList.prepare(atomsObject, suppressDialogs)) {
		deformationGradientChannel->setSize(0);
		strainTensorChannel->setSize(0);
		hydrostaticStrainChannel->setSize(0);
		shearStrainChannel->setSize(0);
		return false;
	}

	// Prepare the output channels.
	deformationGradientChannel->setSize(atomsObject->atomsCount());
	strainTensorChannel->setSize(atomsObject->atomsCount());
	hydrostaticStrainChannel->setSize(atomsObject->atomsCount());
	shearStrainChannel->setSize(atomsObject->atomsCount());

	// The nearest neighbor vectors in primitive lattice space for FCC.
	const Vector3 fccLatticeVectors[12] = {
		Vector3( 1, 0, 0), Vector3( 0, 1, 0), Vector3( 0, 0, 1),
		Vector3(-1, 0, 0), Vector3( 0,-1, 0), Vector3( 0, 0,-1),
		Vector3(-1, 1, 0), Vector3( 0,-1, 1), Vector3(-1, 0, 1),
		Vector3( 1,-1, 0), Vector3( 0, 1,-1), Vector3( 1, 0,-1)
	};
	/// The primitive FCC cell matrix:
	Matrix3 fccPrimitiveCell;
	Matrix3 fccInversePrimitiveCell;
	fccPrimitiveCell.column(0) = Vector3(0.5, 0.5, 0.0);
	fccPrimitiveCell.column(1) = Vector3(0.5, 0.0, 0.5);
	fccPrimitiveCell.column(2) = Vector3(0.0, 0.5, 0.5);
	fccInversePrimitiveCell = fccPrimitiveCell.inverse();

	// This bit array stores for each atom whether it has already been
	// processed.
	QBitArray processedArray(atomsObject->atomsCount());
	// Keep track of the number of atoms that have already been processed during the recursive algorithm.
	int numAtomsProcessed = 0;
	int progressUpdateCounter = 0;

	int numCrystallineAtoms = 0;	// Number of atoms for which the strain has been successfully calculated.
	int numErrorAtoms = 0;			// Number of crystalline atoms for which the strain could not be calculated.
	int numDisorderedAtoms = 0;		// Number of non-crystalline atoms.

	// Iterate over all atoms.
	for(int atomIndex = 0; atomIndex < atomsObject->atomsCount(); atomIndex++) {

		// Skip atoms that have already been processed.
		if(processedArray.at(atomIndex)) continue;

		// Reset output fields for this atom.
		deformationGradientChannel->setTensor2(atomIndex, NULL_MATRIX);
		strainTensorChannel->setSymmetricTensor2(atomIndex, SymmetricTensor2(0));
		hydrostaticStrainChannel->setFloat(atomIndex, 0);
		shearStrainChannel->setFloat(atomIndex, 0);

		// Use only crystalline atoms as starting points for the recursive algorithm.
		if(cnaChannel->getInt(atomIndex) == CommonNeighborAnalysisModifier::FCC) {

			// Gather neighbor vectors of current atom.
			int numNeighbors = 0;
			Vector3 neighborVectors[12];
			int neighborIndices[12];
			for(OnTheFlyNeighborList::iterator neighborIter(neighborList, atomIndex); !neighborIter.atEnd(); neighborIter.next(), numNeighbors++) {
				if(numNeighbors >= 12)
					throw Exception(tr("Found an FCC atom (index %1) with more than 12 nearest neighbors. Please make sure that the cutoff radius used for the CNA modifier is the same as used for the Calculate Intrinsic Strain modifier.").arg(atomIndex));
				neighborVectors[numNeighbors] = neighborIter.delta();
				neighborIndices[numNeighbors] = neighborIter.current();
			}
			if(numNeighbors < 12)
				throw Exception(tr("Found an FCC atom (index %1) with less than 12 nearest neighbors. Please make sure that the cutoff radius used for the CNA modifier is the same as used for the Calculate Intrinsic Strain modifier.").arg(atomIndex));

			// Build a Thompson tetrahedron from the nearest neighbors.
			// All 4 vertices must be neighbors of each other.
			Tensor2 orientation(NULL_MATRIX);
			for(size_t k=0; k<12; k++) {
				size_t vertexIndices[3] = { k, -1, -1 };
				for(size_t i=0; i<12; i++) {
					if(i != k && neighborList.areNeighbors(neighborIndices[i], neighborIndices[k])) {
						vertexIndices[1] = i;
						for(size_t j=0; j<12; j++) {
							if(j != i && j != k && neighborList.areNeighbors(neighborIndices[j], neighborIndices[k]) &&
									neighborList.areNeighbors(neighborIndices[j], neighborIndices[i])) {
								vertexIndices[2] = j;
								break;
							}
						}
						break;
					}
				}
				if(vertexIndices[2] != -1) {
					orientation.column(0) = neighborVectors[vertexIndices[0]];
					orientation.column(1) = neighborVectors[vertexIndices[1]];
					orientation.column(2) = neighborVectors[vertexIndices[2]];
					if(orientation.determinant() < 0.0)
						swap(orientation.column(0), orientation.column(1));
					break;
				}
			}

			// Skip invalid atoms for which no Thompson tetrahedron could be formed.
			if(orientation.determinant() <= FLOATTYPE_EPSILON)
				continue;

			// Store the preliminary orientation matrix for this seed atom in the data channel.
			deformationGradientChannel->setTensor2(atomIndex, orientation);
		}
		else {
			processedArray.setBit(atomIndex);
			numAtomsProcessed++;
			numDisorderedAtoms++;
			continue;
		}

		// This bit array stores for each atom whether it has already been
		// visited during the current run.
		QBitArray visitedArray(atomsObject->atomsCount());
		visitedArray.setBit(atomIndex);
		numAtomsProcessed++;

		// Now recursively iterate over all neighbors of the initial atom and process them
		deque<int> toProcess;
		toProcess.push_back(atomIndex);

		do {
			// Update progress indicator.
			progressUpdateCounter++;
			if((progressUpdateCounter % 4096) == 0) {
				progressUpdateCounter = 0;
				progress.setValue(numAtomsProcessed);
				if(progress.isCanceled()) {
					// Throw away results obtained so far if the user cancels the calculation.
					deformationGradientChannel->setSize(0);
					strainTensorChannel->setSize(0);
					hydrostaticStrainChannel->setSize(0);
					shearStrainChannel->setSize(0);
					return false;
				}
			}

			// Take the next atom to process from the stack.
			int currentAtom = toProcess.front();
			toProcess.pop_front();

			// Look only at crystalline FCC atoms.
			int cnaAtomType = cnaChannel->getInt(currentAtom);
			OVITO_ASSERT(cnaAtomType == CommonNeighborAnalysisModifier::FCC);

			// Gather neighbor vectors of current atom.
			int numNeighbors = 0;
			Vector3 neighborVectors[12];
			int neighborIndices[12];
			for(OnTheFlyNeighborList::iterator neighborIter(neighborList, currentAtom); !neighborIter.atEnd(); neighborIter.next(), numNeighbors++) {
				if(numNeighbors >= 12)
					throw Exception(tr("Found an FCC atom (index %1) with more than 12 nearest neighbors. Please make sure that the cutoff radius used for the CNA modifier is the same as used for the Calculate Intrinsic Strain modifier.").arg(currentAtom));
				neighborVectors[numNeighbors] = neighborIter.delta();
				neighborIndices[numNeighbors] = neighborIter.current();
			}
			if(numNeighbors < 12)
				throw Exception(tr("Found an FCC atom (index %1) with less than 12 nearest neighbors. Please make sure that the cutoff radius used for the CNA modifier is the same as used for the Calculate Intrinsic Strain modifier.").arg(currentAtom));

			// Fetch the preliminary orientation matrix for this atom that has been determined previously.
			Matrix3 orientation = deformationGradientChannel->getTensor2(currentAtom);
			OVITO_ASSERT(orientation.determinant() > FLOATTYPE_EPSILON);

			// Map nearest neighbors to reference lattice.
			Matrix3 inverseOrientation = orientation.inverse();
			unsigned int mapped = 0;

			FloatType level = shearStrainChannel->getFloat(currentAtom);

			Vector3 latticeNeighborVectors[12];
			bool found;
			for(size_t ni=0; ni<12; ni++) {
				Vector3 v = inverseOrientation * neighborVectors[ni];
				Vector3 u;
				u.X = floor(v.X + 0.5);
				u.Y = floor(v.Y + 0.5);
				u.Z = floor(v.Z + 0.5);
				found = false;
				for(size_t nj = 0; nj < 12; nj++) {
					if(u == fccLatticeVectors[nj]) {
						if(mapped & (1 << nj)) break;
						latticeNeighborVectors[ni] = fccLatticeVectors[nj];
						mapped |= (1<<nj);
						found = true;
						break;
					}
				}
				if(!found) break;
			}
			if(found) {
				// Calculate average orientation matrix based on the positions of all nearest neighbors.
				Matrix3 V(NULL_MATRIX);
				Matrix3 W(NULL_MATRIX);
				for(size_t ni=0; ni<12; ni++) {
					const Vector3& nv = neighborVectors[ni];
					const Vector3& lv = latticeNeighborVectors[ni];
					V(0,0) += lv.X * lv.X;
					V(0,1) += lv.Y * lv.X;
					V(0,2) += lv.Z * lv.X;
					V(1,0) += lv.X * lv.Y;
					V(1,1) += lv.Y * lv.Y;
					V(1,2) += lv.Z * lv.Y;
					V(2,0) += lv.X * lv.Z;
					V(2,1) += lv.Y * lv.Z;
					V(2,2) += lv.Z * lv.Z;

					W(0,0) += lv.X * nv.X;
					W(0,1) += lv.Y * nv.X;
					W(0,2) += lv.Z * nv.X;
					W(1,0) += lv.X * nv.Y;
					W(1,1) += lv.Y * nv.Y;
					W(1,2) += lv.Z * nv.Y;
					W(2,0) += lv.X * nv.Z;
					W(2,1) += lv.Y * nv.Z;
					W(2,2) += lv.Z * nv.Z;
				}
				OVITO_ASSERT(V.determinant() > FLOATTYPE_EPSILON);
				OVITO_ASSERT(W.determinant() > FLOATTYPE_EPSILON);

				// Store tensor in output channel.
				orientation = W * V.inverse();
				Tensor2& F = *(deformationGradientChannel->dataTensor2() + currentAtom);
				F = (orientation * fccInversePrimitiveCell) * (1.0/latticeConstant);

				// Calculate strain tensor.
				SymmetricTensor2 strain = (Product_AtA(F) - IDENTITY) * 0.5;
				strainTensorChannel->setSymmetricTensor2(currentAtom, strain);

				// Calculate hydrostatic strain invariant.
				hydrostaticStrainChannel->setFloat(currentAtom, (strain(0,0) + strain(1,1) + strain(2,2)) / 3.0);

				// Calculate shear strain invariant.
				shearStrainChannel->setFloat(currentAtom, sqrt(
						square(strain(0,1)) + square(strain(1,2)) + square(strain(0,2)) +
						(square(strain(1,1) - strain(2,2)) + square(strain(0,0) - strain(2,2)) + square(strain(0,0) - strain(1,1))) / 6.0));

				// Calculate residual
				FloatType residual = 0.0;
				for(size_t ni=0; ni<12; ni++)
					residual += LengthSquared(neighborVectors[ni] - (orientation * latticeNeighborVectors[ni]));
				hydrostaticStrainChannel->setFloat(currentAtom, residual);

				processedArray.setBit(currentAtom);
				numAtomsProcessed++;
				numCrystallineAtoms++;
			}
			else {
				// Reset output fields if we are not able to determine the crystal orientation.
				deformationGradientChannel->setTensor2(currentAtom, NULL_MATRIX);
				strainTensorChannel->setSymmetricTensor2(currentAtom, SymmetricTensor2(0));
				hydrostaticStrainChannel->setFloat(currentAtom, 0);
				shearStrainChannel->setFloat(currentAtom, level);
				continue;
			}
			level++;

			// Put neighbor atoms onto the stack of atoms to be processed.
			for(size_t ni = 0; ni < numNeighbors; ni++) {
				int neighborIndex = neighborIndices[ni];
				if(!visitedArray.at(neighborIndex) && !processedArray.at(neighborIndex)) {
					visitedArray.setBit(neighborIndex);
					if(cnaChannel->getInt(neighborIndex) == CommonNeighborAnalysisModifier::FCC) {
						deformationGradientChannel->setTensor2(neighborIndex, orientation);
						toProcess.push_back(neighborIndex);
					}
					else {
						deformationGradientChannel->setTensor2(neighborIndex, NULL_MATRIX);
						strainTensorChannel->setSymmetricTensor2(neighborIndex, SymmetricTensor2(0));
						hydrostaticStrainChannel->setFloat(neighborIndex, 0);
						shearStrainChannel->setFloat(neighborIndex, 0);
						processedArray.setBit(neighborIndex);
						numAtomsProcessed++;
						numDisorderedAtoms++;
					}
					shearStrainChannel->setFloat(neighborIndex, level);
				}
			}
		}
		while(toProcess.empty() == false);
	}
	numErrorAtoms = atomsObject->atomsCount() - numCrystallineAtoms - numDisorderedAtoms;
	MsgLogger() << "Intrinsic strain calculation statistics:" << endl;
	MsgLogger() << "Number of successfully processed crystalline atoms:" << numCrystallineAtoms << endl;
	MsgLogger() << "Number of crystalline atoms that could not be processed:" << numErrorAtoms << endl;
	MsgLogger() << "Number of non-crystalline atoms:" << numDisorderedAtoms << endl;
	OVITO_ASSERT(numErrorAtoms >= 0);

	return true;
}

IMPLEMENT_PLUGIN_CLASS(CalculateIntrinsicStrainModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void CalculateIntrinsicStrainModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Calculate Intrinsic Strain"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(0,0,0,0);
	layout2->setSpacing(0);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);

	// Lattice constant parameter.
	FloatPropertyUI* latticeConstantPUI = new FloatPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CalculateIntrinsicStrainModifier, _latticeConstant));
	layout2->addWidget(latticeConstantPUI->label(), 0, 0);
	layout2->addWidget(latticeConstantPUI->textBox(), 0, 1);
	layout2->addWidget(latticeConstantPUI->spinner(), 0, 2);
	latticeConstantPUI->setMinValue(0);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	BooleanPropertyUI* saveResultsUI = new BooleanPropertyUI(this, "storeResultsWithScene", tr("Save results in scene file"));
	layout1->addWidget(saveResultsUI->checkBox());

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onCalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Open sub-editors for the result channels.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalculateIntrinsicStrainModifier, deformationGradientChannel), rolloutParams.after(rollout).collapse());
	//new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalculateIntrinsicStrainModifier, strainTensorChannel), rolloutParams.after(rollout).collapse());
	//new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalculateIntrinsicStrainModifier, hydrostaticStrainChannel), rolloutParams.after(rollout).collapse());
	//new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalculateIntrinsicStrainModifier, shearStrainChannel), rolloutParams.after(rollout).collapse());

	// Open a sub-editor for the NearestNeighborList sub-object.
	SubObjectParameterUI* subEditorUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));
}

/******************************************************************************
* Is called when the user presses the "Calculate" button.
******************************************************************************/
void CalculateIntrinsicStrainModifierEditor::onCalculate()
{
	if(!editObject()) return;
	CalculateIntrinsicStrainModifier* modifier = static_object_cast<CalculateIntrinsicStrainModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to analyze atoms."));
		ex.showError();
	}
}

};	// End of namespace CrystalAnalysis
